//
// newslib.c -- library routines for foxynews
//
// Copyright (C) 1999 by Bob Crispen <crispen@hiwaay.net> doing business as
//   Foxy Software
//
// LICENSE:
//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU General Public License as published by
//   the Free Software Foundation; either version 2 of the License, or
//   (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; if not, write to the Free Software
//   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <windows.h>
#include <string.h>
#include <winsock.h>
#include <stdio.h>
#include <ctype.h>
#include <sys/stat.h>
#include <errno.h>
#include <time.h>
#include <malloc.h>
#include <stdlib.h>
#include "foxynews.h"
#include "resource.h"
#include "newslib.h"

// Status returns from NNTP
#define OK_POSTING_OK		200
#define OK_NO_POSTING		201
#define GROUP_SELECTED		211
#define INFORMATION_FOLLOWS	215
#define ARTICLE_FOLLOWS		220
#define XOVER_DATA_FOLLOWS	224
#define AUTH_OK			281
#define MORE_AUTH_REQUIRED	381
#define NO_SUCH_NEWSGROUP	411
#define NO_SUCH_ARTICLE		423
#define AUTH_REQUIRED		480
#define AUTH_FAILED		502

char monthNames[12][3] = {"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul",
    "Aug", "Sep", "Oct", "Nov", "Dec" };

// Child window handles
HWND hwndStatusBar;
HWND hwndEdit2;

int rawbuf_ptr = 0;
int current_line_ptr = 0;
int bytes_rcvd = 0;
unsigned int status;
int num_articles;
unsigned long int article_num;
unsigned long int first;
unsigned long int last;
unsigned long int findPtr;
unsigned long int nMatches;
unsigned long int nAttachments;
unsigned long int nLines;
unsigned long int nBytes;
unsigned long int an;
int this_line;
int article_count = 0;
int subject_field;
int from_field;
int bytes_field;
int lines_field;
int date_field;
int xover_fields;
int hdr_file_line;

char gszTemp[BUFSIZE];
char szResponse[BUFSIZE];
char szSubject[BUFSIZE];
char szMsgDate[BUFSIZE];
char szSender[BUFSIZE];

char sendbuf[BUFSIZE];
char rawbuf[BUFSIZE];
char current_line[BUFSIZE];
char reply[BUFSIZE];
char subj[BUFSIZE];
char filename[BUFSIZE];
char scratchfilename[BUFSIZE];
char boundary[BUFSIZE];
char attfn[BUFSIZE];
char logfn[BUFSIZE];
char savingFileName[BUFSIZE];
char readingHeadersLine[BUFSIZE];

BOOL extracting_attachment;
BOOL extracting_article;
BOOL first_url_found;
BOOL uudecoding;
BOOL content_type_found_in_hdr;
BOOL search_match;
BOOL quit_signal_rcvd = FALSE;
BOOL in_hdr;
BOOL fLoginUsed;
BOOL text_section;

// Testing flags
BOOL xover_enabled = TRUE;

FILE* fd = NULL;
FILE* sfd = NULL;
FILE* afd = NULL;
FILE* arfd = NULL;
FILE* urlfd = NULL;
FILE* logfd = NULL;

time_t dateLimit;

stateMachine sm;

// Arrays for dates, subjects, etc.
char** subject_list;
unsigned long int subject_list_count;
char** newsgroup_list;
unsigned long int newsgroup_list_count;

/*
* Initialize the data that will drive our state machine
*/
void init_state_machine_data(
    HWND	hwndDlg,
    HWND	hwndProgress,
    SOCKET	sockNews,
    char*	szServerName,
    char*	szNewsgroup,
    char*	szSearchString,
    char*	szLoginName,
    char*	szPassword,
    int		nStateCmd,
    int		nDaysPast,
    BOOL	fListNewsgroups,
    BOOL	fShowHdrs,
    BOOL	fPastDays,
    BOOL	fArticleText,
    BOOL	fBinaryAttachments,
    BOOL	fURLs,
    BOOL	fMakeLog,
    BOOL	fDontSaveDups)
{
    int i, sl;

    sm.hwndDlg = hwndDlg;
    sm.hwndProgress = hwndProgress;
    sm.sockNews = sockNews;

    bzero(sm.szServerName, BUFSIZE);
    strcpy(sm.szServerName, szServerName);

    bzero(sm.szNewsgroup, BUFSIZE);
    strcpy(sm.szNewsgroup, szNewsgroup);

    bzero(sm.szLoginName, SHORTBUFSIZE);
    strcpy(sm.szLoginName, szLoginName);

    bzero(sm.szPassword, SHORTBUFSIZE);
    strcpy(sm.szPassword, szPassword);

    bzero(sm.szSearchString, BUFSIZE);
    strcpy(sm.szSearchString, szSearchString);
    sl = strlen(sm.szSearchString);
    for (i=0; i<sl; i++) {
	if (islower(sm.szSearchString[i]))
	    sm.szSearchString[i] = toupper(sm.szSearchString[i]);
	if ((sm.szSearchString[i] == '\r') || (sm.szSearchString[i] == '\n'))
	    sm.szSearchString[i] = '\0';
    }

    sm.nStateCmd = nStateCmd;
    sm.nState = CONNECTING;

    sm.nDaysPast = nDaysPast;
    set_date_limit(sm.nDaysPast);

    sm.fListNewsgroups = fListNewsgroups;
    sm.fShowHdrs = fShowHdrs;
    sm.fPastDays = fPastDays;
    sm.fArticleText = fArticleText;
    sm.fBinaryAttachments = fBinaryAttachments;
    sm.fURLs = fURLs;
    sm.fMakeLog = fMakeLog;
    sm.fDontSaveDups = fDontSaveDups;

    // Other initialization we'll need
    init_buffers();
    hwndStatusBar = GetDlgItem(sm.hwndDlg, IDC_STATUSBAR);
    hwndEdit2 = GetDlgItem(sm.hwndDlg, IDC_EDIT2);
    fLoginUsed = FALSE;
    subject_list_count = 0;
    newsgroup_list_count = 0;
}

/*
* Main state machine that controls the program
* ENTRY:  Data is ready to be read from the news host
* RETURN: state value for next iteration
*/
int state_machine()
{
    if (sm.nState != QUITTING) {
	if (recv_from_news_host(sm.hwndDlg, sm.sockNews)) {
	    sm.nState = QUITTING;
	}
    }
    switch (sm.nState) {
	case CONNECTING:
	    sm.nState = process_connect_reply(sm);
	    break;
	case WAITING_FOR_LIST_NEWSGROUPS_REPLY:
	    sm.nState = process_list_newsgroups_reply(sm);
	    if (sm.nState == LISTING_NEWSGROUPS)
		sm.nState = process_listing_newsgroups(sm);
	    break;
	case WAITING_FOR_LIST_REPLY:
	    sm.nState = process_list_reply(sm);
	    if (sm.nState == LISTING_NEWSGROUPS)
		sm.nState = process_listing_newsgroups(sm);
	    break;
	case LISTING_NEWSGROUPS:
	    if (quit_signal_rcvd) {
		Display("Got quit signal");
		sm.nState = QUITTING;
		break;
	    }
	    sm.nState = process_listing_newsgroups(sm);
	    break;
	case WAITING_FOR_GROUP_REPLY:
	    sm.nState = process_group_reply(sm);
	    break;
	case WAITING_FOR_AUTHINFO_USER_REPLY:
	    sm.nState = process_authinfo_user_reply(sm);
	    if (sm.nState == AUTHORIZATION_FAILED) {
		(void)process_quitting(sm);
	    }
	    break;
	case WAITING_FOR_AUTHINFO_PASS_REPLY:
	    sm.nState = process_authinfo_pass_reply(sm);
	    if (sm.nState == AUTHORIZATION_FAILED) {
		(void)process_quitting(sm);
	    }
	    break;
	case WAITING_FOR_LIST_OVERVIEW_FMT_REPLY:
	    sm.nState = process_list_overview_fmt_reply(sm);
	    if (sm.nState == GETTING_OVERVIEW_FMT)
		sm.nState = process_getting_overview_fmt(sm);
	    break;
	case GETTING_OVERVIEW_FMT:
	    sm.nState = process_getting_overview_fmt(sm);
	    break;
	case WAITING_FOR_XOVER_REPLY:
	    sm.nState = process_xover_reply(sm);
	    if (sm.nState == READING_XOVER_DATA)
		sm.nState = process_reading_xover_data(sm);
	    break;
	case READING_XOVER_DATA:
	    sm.nState = process_reading_xover_data(sm);
	    break;
	case WAITING_FOR_HEAD_REPLY:
	    sm.nState = process_head_reply(sm);
	    if (sm.nState == READING_HEADER)
		sm.nState = process_reading_header(sm);
	    break;
	case READING_HEADER:
	    sm.nState = process_reading_header(sm);
	    break;
	case WAITING_FOR_ARTICLE_REPLY:
	    sm.nState = process_article_reply(sm);
	    if (sm.nState == READING_ARTICLE)
		sm.nState = process_reading_article(sm);
	    break;
	case READING_ARTICLE:
	    sm.nState = process_reading_article(sm);
	    break;
	case QUITTING:
	    sm.nState = process_quitting(sm);
	    break;
    }
    if (sm.nState == QUITTING) {
	sm.nState = process_quitting(sm);
	// Let's be nice and tell the user that the login really wasn't
	// required
	if (!fLoginUsed && (sm.szLoginName[0] != '\0')) {
	    fLoginUsed = TRUE;
	    return LOGIN_NOT_REQUIRED;
	}
    }
    return sm.nState;
}

/************************* MAIN PROCESS ROUTINES **************************/

/*
* Connect to the news server.
* ENTRY: Notification has been received that data is ready to read
*  from the socket connect to the news server.
* This executes once.
*/
int process_connect_reply(stateMachine s)
{
    int st;
    struct stat ss;

    st = get_one_line(szResponse);
    if (st < 0)
	return QUITTING;
    bzero(reply, BUFSIZE);
    if (2 > (sscanf(szResponse, "%u %[^\r\n]", &status, reply))) {
	MessageBox(s.hwndDlg,
	    "Can't get status in process connecting",
	    "Message parse error", MB_OK);
	return QUITTING;
    }
    if ((status != OK_POSTING_OK) && (status != OK_NO_POSTING)) {
	sprintf(gszTemp, "Status = %u\r\n", status);
	MessageBox(s.hwndDlg,
	    gszTemp,
	    "News server not ready", MB_OK);
	return QUITTING;
    }

    // The news host is there.  Let's create the directory for the
    // news host if it isn't already there.
    if ((stat(s.szServerName, &ss)) != 0) {
	if (!CreateDirectory(s.szServerName, NULL)) {
	    sprintf(gszTemp, "Can't create directory\r\n%s", s.szServerName);
	    MessageBox(s.hwndDlg,
		gszTemp,
		"CreateDirectory() error", MB_OK);
	    return QUITTING;
	}
    }

    if (s.fListNewsgroups) {
	Display3("Listing newsgroups");
	bzero(sendbuf, BUFSIZE);
	sprintf(sendbuf, "LIST NEWSGROUPS\r\n");
	st = send(s.sockNews, sendbuf, strlen(sendbuf), 0);
	return WAITING_FOR_LIST_NEWSGROUPS_REPLY;
    } else {
	sprintf(gszTemp, "Getting data for %s", s.szNewsgroup);
	Display3(gszTemp);
	bzero(sendbuf, BUFSIZE);
	sprintf(sendbuf, "GROUP %s\r\n", s.szNewsgroup);
	st = send(s.sockNews, sendbuf, strlen(sendbuf), 0);
	return WAITING_FOR_GROUP_REPLY;
    }
}

/*
* Process the reply from a LIST NEWSGROUPS command
* ENTRY: Notification has been received that data is ready to read
* This executes once.
*/
int process_list_newsgroups_reply(stateMachine s)
{
    int st;

    if ((st = get_one_line(szResponse)) < 0)
	return QUITTING;

    // Get the status
    bzero(reply, BUFSIZE);
    if (2 > (sscanf(szResponse, "%u %[^\r\n]", &status, reply))) {
	MessageBox(s.hwndDlg,
	    "Can't read list newsgroups reply",
	    "Message parse error", MB_OK);
	return QUITTING;
    }
    if (status == AUTH_REQUIRED) {
	if (s.szLoginName[0] == '\0') {
	    MessageBox(s.hwndDlg,
		"Please check \"Server requires login\" and try again",
		"Authorization Error", MB_OK | MB_ICONEXCLAMATION);
	    return QUITTING;
	}
	fLoginUsed = TRUE;
	bzero(sendbuf, BUFSIZE);
	sprintf(sendbuf, "AUTHINFO USER %s\r\n", s.szLoginName);
	st = send(s.sockNews, sendbuf, strlen(sendbuf), 0);
	return WAITING_FOR_AUTHINFO_USER_REPLY;
    }
    if (status != INFORMATION_FOLLOWS) {
	// Try a plain LIST instead
	bzero(sendbuf, BUFSIZE);
	sprintf(sendbuf, "LIST\r\n");
	st = send(s.sockNews, sendbuf, strlen(sendbuf), 0);
	return WAITING_FOR_LIST_REPLY;
    }

    // Clear out the header display window
    SendMessage(hwndEdit2, EM_SETSEL, 0, -1);
    SendMessage(hwndEdit2, WM_CLEAR, 0, 0);
    SendMessage(hwndEdit2, EM_SETSEL, 0, 0);
    Display("Listing newsgroups");
    return LISTING_NEWSGROUPS;
}

/*
* Process the reply from a LIST command
* ENTRY: Notification has been received that data is ready to read
* This executes once.
*/
int process_list_reply(stateMachine s)
{
    int st;

    if ((st = get_one_line(szResponse)) < 0)
	return QUITTING;

    // Get the status back
    bzero(reply, BUFSIZE);
    if (2 > (sscanf(szResponse, "%u %[^\r\n]", &status, reply))) {
	MessageBox(s.hwndDlg,
	    "Can't read list newsgroups reply",
	    "Message parse error", MB_OK);
	return QUITTING;
    }
    if (status != INFORMATION_FOLLOWS) {
	sprintf(gszTemp,
	    "Tried LIST NEWSGROUPS and LIST\r\nStatus = %d", status);
	MessageBox(s.hwndDlg,
	    gszTemp,
	    "List error", MB_OK);
	return QUITTING;
    }

    // Clear out the header display window
    SendMessage(hwndEdit2, EM_SETSEL, 0, -1);
    SendMessage(hwndEdit2, WM_CLEAR, 0, 0);
    SendMessage(hwndEdit2, EM_SETSEL, 0, 0);
    Display("Listing newsgroups");
    return LISTING_NEWSGROUPS;
}

/*
* Get the list of newsgroups and display them in the window
* ENTRY: Status reply has been received from a LIST NEWSGROUPS command.
* This is executed repeatedly until all the newsgroup information has been
*  read or a fatal error has occurred.
*/
int process_listing_newsgroups(stateMachine s)
{
    int st, i;

    while ((st = get_one_line(szResponse)) > 0) {
	if (newsgroup_list_count == 0) {
	    newsgroup_list = (char**)malloc(1024 * sizeof(char*));
	    if (newsgroup_list == NULL) {
		MessageBox(s.hwndDlg,
		    "malloc() failed",
		    "Virtual memory error", MB_OK);
		return QUITTING;
	    }
	} else if (newsgroup_list_count % 1024 == 0) {
	    newsgroup_list = (char**)realloc(
		(void*)newsgroup_list, 1024 * sizeof(char*));
	    if (newsgroup_list == NULL) {
		MessageBox(s.hwndDlg,
		    "Array expansion failed",
		    "Virtual memory error", MB_OK);
		return QUITTING;
	    }
	}
	newsgroup_list[newsgroup_list_count] =
	    (char*)malloc(strlen(szResponse)+1);
	if (newsgroup_list[newsgroup_list_count] == NULL) {
	    MessageBox(s.hwndDlg,
		"Add string to array failed",
		"Virtual memory error", MB_OK);
	    return QUITTING;
	}
	bzero(newsgroup_list[newsgroup_list_count], strlen(szResponse)+1);
	strcpy(newsgroup_list[newsgroup_list_count++], szResponse);
    }
    if (st == 0)
	return LISTING_NEWSGROUPS;

    // Sort the list of newsgroups and display it
    qsort(newsgroup_list, newsgroup_list_count, sizeof(char*),
	compare_newsgroups);
    for (i=0; i<newsgroup_list_count; i++)
	Display2(newsgroup_list[i]);
    for (i=0; i<newsgroup_list_count; i++)
	free(newsgroup_list[i]);
    free(newsgroup_list);
    return QUITTING;
}

/*
* Get data about the newsgroup and update the header file if needed.
*  If the command is "Get Articles", don't bother to update the header file.
* ENTRY: Notification has been received that data is ready to read
* This executes once.
*/
int process_group_reply(stateMachine s)
{
    int st;
    unsigned long int lFirst, lLast, lNumArticles;
    char *sz;
    struct stat ss;

    if ((st = get_one_line(szResponse)) < 0)
	return QUITTING;

    bzero(reply, BUFSIZE);
    status = 0;
    if (5 > (sscanf(szResponse, "%u %u %u %u %[^\r\n]",
	&status, &num_articles, &first, &last, reply))) {
	if (status == NO_SUCH_NEWSGROUP) {
	    MessageBox(s.hwndDlg,
		"No such newsgroup",
		"Newsgroup error", MB_OK);
	    return QUITTING;
	}
	if (status == AUTH_REQUIRED) {
	    if (s.szLoginName[0] == '\0') {
		MessageBox(s.hwndDlg,
		    "Please check \"Server requires login\" and try again",
		    "Authorization Error", MB_OK | MB_ICONEXCLAMATION);
		return QUITTING;
	    }
	    fLoginUsed = TRUE;
	    bzero(sendbuf, BUFSIZE);
	    sprintf(sendbuf, "AUTHINFO USER %s\r\n", s.szLoginName);
	    st = send(s.sockNews, sendbuf, strlen(sendbuf), 0);
	    return WAITING_FOR_AUTHINFO_USER_REPLY;
	}
	if (status == 0) {
	    MessageBox(s.hwndDlg,
		"Can't get status in getting newsgroup data",
		"Message parse error", MB_OK);
	    return QUITTING;
	}
	sprintf(gszTemp, "Status = %u");
	MessageBox(s.hwndDlg,
	    gszTemp,
	    "Message parse error", MB_OK);
	return QUITTING;
    }
    /*
    * If we're extracting attachments, let's see if the log file exists.
    * If it does, just open it for reading, and don't bother to update
    * the header data in the log file from the server.  If it doesn't exist,
    * fall through and create it and get the latest header data from the
    * server.
    */
    if (s.nStateCmd == GET_ATTACHMENTS) {
	bzero(filename, BUFSIZE);
	strcpy(filename, s.szServerName);
	filename[strlen(filename)] = '\\';
	strcat(filename, s.szNewsgroup);
	strcat(filename, ".log");
	if ((stat(filename, &ss)) == 0) {
	    fd = fopen(filename, "rb");
	    if ((set_up_for_reading_articles(s)) == -1) {
		return QUITTING;
	    }
	    return WAITING_FOR_ARTICLE_REPLY;
	}
    }

    sprintf(gszTemp, "Server has %u articles from %u to %u",
	num_articles, first, last);
    Display(gszTemp);
    /*
    * Open a file called <newshost>\<newsgroup>.log
    * If the file or directory don't exist, create them.
    */
    if ((stat(s.szServerName, &ss)) == 0) {
	/* Directory exists */
	if (ss.st_mode & S_IFDIR) {
	    bzero(filename, BUFSIZE);
	    strcpy(filename, s.szServerName);
	    filename[strlen(filename)] = '\\';
	    strcat(filename, s.szNewsgroup);
	    strcat(filename, ".log");
	    if ((stat(filename, &ss)) == 0) {
		/* Header file exists in that directory */
		fd = fopen(filename, "rb");
		if (fd == 0) {
		    MessageBox(s.hwndDlg,
			"Can't open header file for read",
			"fopen() error", MB_OK);
		    return QUITTING;
		}
		/*
		* Since the file already exists, we can save quite a bit
		* of time by not having to read all the headers from the
		* news host.  The file probably has old data in it, so
		* first we need to update the file from the news server.
		*/
		bzero(subj, BUFSIZE);
		sz = fgets(subj, BUFSIZE, fd);
		if (sz == NULL) {
		    MessageBox(s.hwndDlg,
			"Can't fetch first line from header file",
			"File read error", MB_OK);
		    return QUITTING;
		}
		if (3 > (sscanf(subj, "%u %u %u",
		    &lNumArticles, &lFirst, &lLast))) {
		    MessageBox(s.hwndDlg,
			"Can't parse first line from header file",
			"Message parse error", MB_OK);
		    return QUITTING;
		}
		/*
		* First, see if the news server has rebuilt its spool.
		* If it has, forget saving anything from the file.
		* Likewise if we aren't reading all the headers, start over
		*/
		if (first < lFirst) {
		    Display("News server has rebuilt its spool.  Fetching all headers");
		    fclose(fd);
		    fd = fopen(filename, "wb");
		    if (fd == 0) {
			MessageBox(s.hwndDlg,
			    "Can't create header file",
			    "fopen() error", MB_OK);
			return QUITTING;
		    }
		    lLast = first;
		    fprintf(fd, "%u %u %u\r\n", num_articles, first, last);
		}
		/*
		* Now expire any old articles
		*/
		if (lFirst < first) {
		    sprintf(gszTemp, "Expiring %u articles", first-lFirst);
		    Display(gszTemp);
		    if ((copy_header_file(s.hwndDlg)) == -1) {
			return QUITTING;
		    }
		}
		/*
		* If new headers are present on the news server, read them
		*/
		if (last > lLast) {
		    bzero(readingHeadersLine, BUFSIZE);
		    sprintf(readingHeadersLine, "Reading %u new headers",
			last-lLast);
		    Display(readingHeadersLine);
		    if ((copy_header_file(s.hwndDlg)) == -1) {
			return QUITTING;
		    }
		    fclose(fd);
		    fd = fopen(filename, "ab");
		    if (fd == 0) {
			MessageBox(s.hwndDlg,
			    "Can't open header file for append",
			    "fopen() error", MB_OK);
			return QUITTING;
		    }
		    first = lLast + 1;
		} else {
		    Display("No new headers on server");
		    if (s.fShowHdrs) {
			list_headers(
			    s.hwndDlg, s.hwndProgress, s.fPastDays,
			    s.szSearchString);
			return QUITTING;
		    } else {
			if ((set_up_for_reading_articles(s)) == -1) {
			    return QUITTING;
			}
			return WAITING_FOR_ARTICLE_REPLY;
		    }
		}
	    } else {
		/* Create the header file */
		bzero(readingHeadersLine, BUFSIZE);
		sprintf(readingHeadersLine, "Building new file of %u headers",
		    num_articles);
		Display(readingHeadersLine);
		fd = fopen(filename, "wb");
		if (fd == 0) {
		    MessageBox(s.hwndDlg,
			"Can't create header file",
			"fopen() error", MB_OK);
		    return QUITTING;
		}
		fprintf(fd, "%u %u %u\r\n", num_articles, first, last);
	    }
	} else {
	    /* File exists, but isn't a directory */
	    MessageBox(s.hwndDlg,
		"File with newsgroup name exists, but isn't a directory",
		"fopen() error", MB_OK);
	    return QUITTING;
	}
    } else {
	/* Create the directory */
	sprintf(gszTemp, "Building new file of %u headers", num_articles);
	Display(gszTemp);
	if (!CreateDirectory(s.szServerName, NULL)) {
	    MessageBox(s.hwndDlg,
		"Can't create directory",
		"fopen() error", MB_OK);
	    return QUITTING;
	}
	bzero(filename, BUFSIZE);
	strcpy(filename, s.szServerName);
	filename[strlen(filename)] = '\\';
	strcat(filename, s.szNewsgroup);
	strcat(filename, ".log");
	/* Create the header file */
	fd = fopen(filename, "wb");
	if (fd == 0) {
	    MessageBox(s.hwndDlg,
		"Can't create header file",
		"fopen() error", MB_OK);
	    return QUITTING;
	}
	fprintf(fd, "%u %u %u\r\n", num_articles, first, last);
    }
    article_num = first;
    if (quit_signal_rcvd) {
	Display("Got quit signal");
	return QUITTING;
    }

    // Send the LIST OVERVIEW.FMT message and enter GETTING_OVERVIEW_FMT
    // state.
    if (xover_enabled) {
	xover_fields = 0;
	subject_field = 0;
	from_field = 0;
	bytes_field = 0;
	lines_field = 0;
	date_field = 0;
	sprintf(sendbuf, "LIST OVERVIEW.FMT\r\n");
	st = send(s.sockNews, sendbuf, strlen(sendbuf), 0);
	Display3("Getting overview format");
	return WAITING_FOR_LIST_OVERVIEW_FMT_REPLY;
    } else {
	sprintf(sendbuf, "HEAD %u\r\n", article_num);
	st = send(s.sockNews, sendbuf, strlen(sendbuf), 0);
	Display3("Reading new headers (slow method)");
	return WAITING_FOR_HEAD_REPLY;
    }
}

/*
* Check the message returned from the LIST OVERVIEW.FMT command.
* ENTRY: Notification has been received that data is ready to read
* This is executed once.
*/
int process_list_overview_fmt_reply(stateMachine s)
{
    int st;

    if ((st = get_one_line(szResponse)) < 0)
	return QUITTING;

    bzero(reply, BUFSIZE);
    if (2 > (sscanf(szResponse, "%u %[^\r\n]", &status, reply))) {
	MessageBox(s.hwndDlg,
	    "Can't read LIST OVERVIEW.FMT reply",
	    "Message parse error", MB_OK);
	return QUITTING;
    }
    if (status != INFORMATION_FOLLOWS) {
	// Forget XOVER and issue one HEAD at a time
	sprintf(sendbuf, "HEAD %u\r\n", article_num);
	st = send(s.sockNews, sendbuf, strlen(sendbuf), 0);
	Display3("Reading new headers (slow method)");
	return WAITING_FOR_HEAD_REPLY;
    }
    return GETTING_OVERVIEW_FMT;
}

/*
* Process the data from a LIST OVERVIEW.FMT command
* ENTRY: Notification has been received that data is ready to read
* This is executed repeatedly until all the OVERVIEW.FMT data is received
* or a fatal error occurs.
*/
int process_getting_overview_fmt(stateMachine s)
{
    int st;

    while ((st = get_one_line(szResponse)) > 0) {
	// Process the data fields, one per line
	xover_fields++;
	if (!strncmp(szResponse, "Subject:", 8))
	    subject_field = xover_fields;
	if (!strncmp(szResponse, "From:", 5))
	    from_field = xover_fields;
	if (!strncmp(szResponse, "Date:", 5))
	    date_field = xover_fields;
	if (!strncmp(szResponse, "Bytes:", 6))
	    bytes_field = xover_fields;
	if (!strncmp(szResponse, "Lines:", 6))
	    lines_field = xover_fields;
    }
    if (st == 0) {
	return GETTING_OVERVIEW_FMT;
    }
    // We have to have at least a Subject: and Date:
    if ((subject_field !=0) && (date_field !=0)) {
	sprintf(sendbuf, "XOVER %u-%u\r\n", first, last);
	st = send(s.sockNews, sendbuf, strlen(sendbuf), 0);
	Display3("Reading new headers (fast method)");
	return WAITING_FOR_XOVER_REPLY;
    } else {
	sprintf(sendbuf, "HEAD %u\r\n", article_num);
	st = send(s.sockNews, sendbuf, strlen(sendbuf), 0);
	Display3("Reading new headers (slow method)");
	return WAITING_FOR_HEAD_REPLY;
    }
}

/*
* Process the reply from an XOVER command
* ENTRY: Notification has been received that data is ready to read
* This is executed once.
*/
int process_xover_reply(stateMachine s)
{
    int st;

    if ((st = get_one_line(szResponse)) < 0)
	return QUITTING;

    bzero(reply, BUFSIZE);
    if (2 > (sscanf(szResponse, "%u %[^\r\n]", &status, reply))) {
	MessageBox(s.hwndDlg,
	    "Can't read XOVER DATA reply",
	    "Message parse error", MB_OK);
	return QUITTING;
    }
    if (status != XOVER_DATA_FOLLOWS) {
	// Forget XOVER and issue one HEAD at a time
	sprintf(sendbuf, "HEAD %u\r\n", article_num);
	st = send(s.sockNews, sendbuf, strlen(sendbuf), 0);
	Display3("Reading new headers (slow method)");
	return WAITING_FOR_HEAD_REPLY;
    }
    nLines = 0;
    return READING_XOVER_DATA;
}

/*
* Get header data using the XOVER method.
* ENTRY: Notification has been received that data is ready to read
* This is executed repeatedly until all the header information has been
*  read or a fatal error has occurred.
*/
int process_reading_xover_data(stateMachine s)
{
    int i, j;
    int st;
    int disp;

    while ((st = get_one_line(szResponse)) > 0) {
	if (last != first) {
	    disp = (100*(article_num-first))/(last-first);
	    if (disp > 100)
		disp = 100;
	    SendMessage(s.hwndProgress, PBM_SETPOS, (WPARAM)disp, 0L);
	    sprintf(gszTemp, "%s %d%%", readingHeadersLine, disp);
	    Display(gszTemp);
	}
	if (1 < (sscanf(szResponse, "%u ", &an))) {
	    MessageBox(s.hwndDlg,
		"Error reading article number from xover message",
		"Message parse error", MB_OK);
	    return QUITTING;
	}
	j = 0;
	for (i=1; i<xover_fields; i++) {
	    for (; szResponse[j] != '\t'; j++);
	    j++;
	    if (i == from_field) {
		get_sender(&szResponse[j], TRUE, FALSE);
	    }
	    if (i == date_field) {
		get_date(&szResponse[j]);
	    }
	    if (i == bytes_field) {
		if (1 < (sscanf(&szResponse[j], "%u", &nBytes))) {
		    MessageBox(s.hwndDlg,
			"Error reading number of bytes from xover message",
			"Message parse error", MB_OK);
		    return QUITTING;
		}
	    }
	    if (i == lines_field) {
		if (1 < (sscanf(&szResponse[j], "%u", &nLines))) {
		    MessageBox(s.hwndDlg,
			"Error reading number of lines from xover message",
			"Message parse error", MB_OK);
		    return QUITTING;
		}
	    }
	    if (i == subject_field) {
		get_subject(&szResponse[j], TRUE, FALSE);
	    }
	}
	fprintf(fd, "%u\t%u\t%u\t%s\t%s\t%s\r\n",
	    an, nLines, nBytes, szMsgDate, szSender, szSubject);
	article_num++;
    }
    if (st == 0)
	return READING_XOVER_DATA;
    if (quit_signal_rcvd) {
	Display("Got quit signal");
	return QUITTING;
    }
    if (s.fShowHdrs) {
	list_headers(s.hwndDlg, s.hwndProgress, s.fPastDays, s.szSearchString);
	return QUITTING;
    } else {
	if ((set_up_for_reading_articles(s)) == -1) {
	    return QUITTING;
	}
	return WAITING_FOR_ARTICLE_REPLY;
    }
}

/*
* Process the reply from a HEAD command
* ENTRY: Notification has been received that data is ready to read
* This is executed once for each header.
*/
int process_head_reply(stateMachine s)
{
    int st;
    BOOL subject_line_found = FALSE;

    if ((st = get_one_line(szResponse)) < 0)
	return QUITTING;

    nLines = 0;
    status = 0;
    an = 0;
    if (2 > (sscanf(szResponse, "%u %u", &status, &an))) {
	if (status == NO_SUCH_ARTICLE) {
	    /* Not an error */
	    return next_header(s);
	}
	if (status > 399) {
	    sprintf(gszTemp, "Status = %u", status);
	    Display(gszTemp);
	    return QUITTING;
	}
	// Screwed up header on the server.  Skip it.
	return next_header(s);
    }
    return READING_HEADER;
}

/*
* Get header data using the HEAD method.
* ENTRY: The status from the HEAD command has been processed
* This is executed repeatedly until all the header lines have been read
*  or a fatal error has occurred.
*/
int process_reading_header(stateMachine s)
{
    int st;
    BOOL subject_line_found = FALSE;
    int disp;

    while ((st = get_one_line(szResponse)) > 0) {
	if (!strncmp(szResponse, "Lines:", 6)) {
	    if (1 < (sscanf(&szResponse[7], "%u", &nLines))) {
		MessageBox(s.hwndDlg,
		    "Error reading number of lines from header",
		    "Message parse error", MB_OK);
		return QUITTING;
	    }
	}
	if (!strncmp(szResponse, "Bytes:", 6)) {
	    if (1 < (sscanf(&szResponse[7], "%u", &nBytes))) {
		MessageBox(s.hwndDlg,
		    "Error reading number of bytes from header",
		    "Message parse error", MB_OK);
		return QUITTING;
	    }
	}
	if (!strncmp(szResponse, "Date:", 5)) {
	    get_date(&szResponse[5]);
	}
	if (!strncmp(szResponse, "From:", 5)) {
	    get_sender(&szResponse[5], FALSE, FALSE);
	}
	if (!strncmp("Subject:", szResponse, 8)) {
	    get_subject(&szResponse[8], FALSE, FALSE);
	    subject_line_found = TRUE;
	} else {
	    // Check for continued subject line
	    if (subject_line_found) {
		if (isspace(szResponse[0])) {
		} else {
		    subject_line_found = FALSE;
		}
	    }
	}
    }
    if (st == 0) {
	return READING_HEADER;
    } else {
	if (quit_signal_rcvd) {
	    Display("Got quit signal");
	    return QUITTING;
	}
	fprintf(fd, "%u\t%u\t%u\t%s\t%s\t%s\r\n",
	    an, nLines, nBytes, szMsgDate, szSender, szSubject);
	if (last != first) {
	    disp = (100*(article_num-first))/(last-first);
	    if (disp > 100)
		disp = 100;
	    SendMessage(s.hwndProgress, PBM_SETPOS, (WPARAM)disp, 0L);
	}
	return next_header(s);
    }
}

/*
* Get reply to ARTICLE command
* ENTRY: Notification has been received that data is ready to read
* This executes once.
*/
int process_article_reply(stateMachine s)
{
    int st;
    FINDTEXT ft;
    unsigned long int cp;
    CHARRANGE cr;

    if ((st = get_one_line(szResponse)) < 0)
	return QUITTING;

    if (1 > (sscanf(szResponse, "%u ", &status))) {
	MessageBox(s.hwndDlg,
	    "Can't read status in get body data",
	    "Message parse error", MB_OK);
	return QUITTING;
    }
    if (status != ARTICLE_FOLLOWS) {
	if (status != NO_SUCH_ARTICLE) {
	    sprintf(gszTemp, "Article Status = %u", status);
	    MessageBox(s.hwndDlg,
		gszTemp,
		"Message parse error", MB_OK);
	    return QUITTING;
	}
	for (;;) {
	    an = get_hdr_file_line_and_send_body(
		s.hwndDlg, s.hwndProgress, s.sockNews, s.szSearchString);
	    if (an == -1)
		return QUITTING;
	    if (search_match)
		break;
	}
	return WAITING_FOR_ARTICLE_REPLY;
    }

    // Highlight the article in the edit window
    ft.chrg.cpMin = findPtr;
    ft.chrg.cpMax = -1;
    ft.lpstrText = szSubject;

    // Clicking on the window or cutting and pasting can mess up
    // the NOHIDESEL flag.  Make sure it's set.
    SendMessage(hwndEdit2, EM_SETOPTIONS, (WPARAM)ECOOP_OR,
	(LPARAM)(ECO_NOHIDESEL | ECO_AUTOVSCROLL));

    // Find the subject in the window
    cp = SendMessage(hwndEdit2, EM_FINDTEXT, 0, (LPARAM)&ft);
    cr.cpMin = cp;
    cr.cpMax = cp+strlen(szSubject);
    findPtr = cr.cpMax;

    // Select the message and scroll so you can see it
    SendMessage(hwndEdit2, EM_EXSETSEL, 0, (LPARAM)&cr);
    SendMessage(hwndEdit2, EM_SCROLLCARET, 0, 0);

    // But turn off AUTOVSCROLL when you've done that
    SendMessage(hwndEdit2, EM_SETOPTIONS, (WPARAM)ECOOP_XOR,
	(LPARAM)ECO_AUTOVSCROLL);

    if (s.fArticleText) {
	if (!extracting_article) {
	    if ((open_article_file(s.hwndDlg, s.szNewsgroup)) == -1) {
		return QUITTING;
	    }
	}
    }
    this_line = 0;
    return READING_ARTICLE;
}

/*
* Get article data
* ENTRY: Notification has been received that data is ready to read
* This is executed repeatedly until all the lines have been read
*  or a fatal error has occurred.
*/
int process_reading_article(stateMachine s)
{
    int i, st;
    int permissions;
    char afn[BUFSIZE];
    char beginstring[BUFSIZE];
    int disp;
    char* sp;
    char* sp2;

    while ((st = get_one_line(szResponse)) > 0) {

	// If we're printing article text, print the header lines too
	if (s.fArticleText) {
	    fprintf(arfd, "%s", szResponse);
	}

	// Do these things in the headers and the bodies
	if (s.fURLs) {
	    // XXX -- one URL per line for now
	    sp = strstr(szResponse, "http://");
	    if (!sp)
		sp = strstr(szResponse, "ftp://");
	    if (!sp)
		sp = strstr(szResponse, "file://");
	    if (sp) {
		if (!urlfd) {
		    MessageBox(s.hwndDlg,
			"URL file not open\n",
			"Program bug", MB_OK);
		    return QUITTING;
		}
		if (!first_url_found) {
		    Display("Saving URL");
		    fprintf(urlfd, "<LI>From: %s\n", szSender);
		    fprintf(urlfd, "<BR>Date: %s\n", szMsgDate);
		    fprintf(urlfd, "<BR>Subject: %s\n", szSubject);
		    fprintf(urlfd, "<P>\n<UL>\n");
		    first_url_found = TRUE;
		}
		bzero(gszTemp, BUFSIZE);
		sscanf(sp, "%s", gszTemp);
		// Now clean up the URL
		if (sp = strchr(gszTemp, '"')) {
		    for (; *sp != '\0'; sp++) {
			*sp = '\0';
		    }
		}
		for (i=strlen(gszTemp); isspace(gszTemp[i]); --i)
		    gszTemp[i] = '\0';
		if (sp = strrchr(gszTemp, '/')) {
		    for (; (!ispunct(*sp) || (*sp == '.') || (*sp == '/') ||
			(*sp == '#') || (*sp == '?') || (*sp == '~')); sp++);
		    *sp = '\0';
		}
		fprintf(urlfd, "<LI> <A HREF=\"%s\">%s</A>\n",
		    gszTemp, gszTemp);
	    }
	}
	this_line++;

	// Update the progress bar and percentage display
	if (nLines) {
	    disp = (100*this_line)/nLines;
	    if (disp > 100)
		disp = 100;
	    SendMessage(s.hwndProgress, PBM_SETPOS, (WPARAM)disp, 0L);
	    sprintf(gszTemp, "%s %d%%", savingFileName, disp);
	    Display(gszTemp);
	}

	// Do these things only in the header
	if (in_hdr) {

	    // End of header marked with blank line
	    if (szResponse[0] == '\r') {
		in_hdr = FALSE;
	    }

	    // If there's a Content-Type field in the header, there may
	    // be a MIME attachment
	    if (!strncmp(szResponse, "Content-Type:", 13)) {
		content_type_found_in_hdr =
		    ((strstr(szResponse, "multipart/mixed") != NULL) ||
		    (strstr(szResponse, "multipart/alternative") != NULL) ||
		    (strstr(szResponse, "multipart/related") != NULL));
	    }

	    // If there's a MIME attachment, get the boundary string
	    // which should always follow "Content-Type:"
	    if (content_type_found_in_hdr) {
		if (sp = strstr(szResponse, "boundary=")) {
		    bzero(boundary, sizeof(boundary));
		    sp = strchr(sp, '\"');
		    if (sp) {
			sp2 = strchr(++sp, '\"');
			if (sp2) {
			    *sp2 = '\0';
			    strcpy(boundary, sp);
			}
		    }
		}
	    }

	} else {

	// Do these things only the body
	    if (s.fBinaryAttachments) {

		// Look for a "begin" marking a uuencoded attachment
		bzero(afn, BUFSIZE);
		bzero(beginstring, BUFSIZE);
		if (3 == (sscanf(szResponse, "%s %u %[^\r\n]",
		    beginstring, &permissions, afn)))
		{
		    if (!strcmp(beginstring, "begin")) {
			// It's a uuencoded attachment
			clean_up_filename(afn);
			if ((open_attachment_file(
			    s.hwndDlg, s.szNewsgroup, afn, s.fMakeLog,
			    s.fDontSaveDups)) == -1)
			    return QUITTING;
			uudecoding = TRUE;
		    }
		}

		// Do this stuff only if there's potentially a MIME attachment
		// in the body.
		if (content_type_found_in_hdr) {
		    // The next boundary string marks the end of the current
		    // section.
		    if ((boundary[0] != '\0') &&
			(strstr(szResponse, boundary)))
		    {
			// If you're already writing the file, when you find
			// another boundary, stop
			if (extracting_attachment) {
			    extracting_attachment = FALSE;
			    if (afd)
				fclose(afd);
			    Display("done");
			    SendMessage(s.hwndProgress, PBM_SETPOS,
				(WPARAM)0, 0L);
			    bzero(savingFileName, BUFSIZE);
			    sprintf(savingFileName, "Scanning article");
			}
		    }
		    // Some MIME-encoded messages change boundaries in
		    // the message body
		    if (sp = strstr(szResponse, "boundary=")) {
			bzero(boundary, sizeof(boundary));
			sp = strchr(sp, '"');
			if (sp) {
			    sp2 = strchr(++sp, '"');
			    if (sp2) {
				*sp2 = '\0';
				strcpy(boundary, sp);
			    }
			}
		    }
		    // We may be able to begin another MIME section
		    if (!strncmp(szResponse, "Content-Type:", 13)) {
			// Ignore text and multipart sections.  Anything
			// else, we'll try to save.
			text_section =
			    (((strstr(szResponse, "text/")) != NULL) ||
			    ((strstr(szResponse, "multipart/")) != NULL));
		    }
		    // We needn't worry about a uuencoded section either.
		    // We'll pick it up above.
		    if (!strncmp(szResponse, "Content-Transfer-Encoding:",26)) {
			text_section =
			    ((strstr(szResponse, "x-uuencode")) != NULL);
		    }
		    // If we aren't in a text section and we find a
		    // "filename=" string, grab the filename and open the file.
		    if ((sp = (strstr(szResponse, "filename="))) &&
			!extracting_attachment && !text_section)
		    {
			sp2 = strrchr(&sp[10], '"');
			if (sp2) {
			    *sp2 = '\0';
			    strcpy(afn, &sp[10]);
			    clean_up_filename(afn);
			    if ((open_attachment_file(
				s.hwndDlg, s.szNewsgroup, afn, s.fMakeLog,
				s.fDontSaveDups)) == -1)
				return QUITTING;
			}
		    }
		}

		if (extracting_attachment && !in_hdr) {
		    if (uudecoding) {
			uudecode(szResponse, afd);
		    } else {
			(void)b64decode(szResponse, afd, (char*)NULL);
		    }
		}
		// Check for the "end" statement
		if (uudecoding && extracting_attachment) {
		    if (!strcmp(szResponse, "end\r\n")) {
			extracting_attachment = FALSE;
			uudecoding = FALSE;
			Display("done");
			if (afd)
			    fclose(afd);
		    }
		}
	    }
	}
    }
    if (st == 0) {
	return READING_ARTICLE;
    }
    if (st == -1) {
	if (quit_signal_rcvd) {
	    Display("Got quit signal");
	    return QUITTING;
	}
	if (extracting_attachment) {
	    extracting_attachment = FALSE;
	    uudecoding = FALSE;
	    Display("done");
	    if (afd)
		fclose(afd);
	}
	if (extracting_article) {
	    extracting_article = FALSE;
	    Display("done");
	    if (arfd)
		fclose(arfd);
	}
	if (first_url_found) {
	    first_url_found = FALSE;
	    Display("done");
	    fprintf(urlfd, "</UL>\n<P>\n");
	}
	SendMessage(s.hwndProgress, PBM_SETPOS, (WPARAM)0, 0L);
	bzero(savingFileName, BUFSIZE);
	sprintf(savingFileName, "Scanning article");
    }
    for (;;) {
	an = get_hdr_file_line_and_send_body(
	    s.hwndDlg, s.hwndProgress, s.sockNews, s.szSearchString);
	if (an == -1)
	    return QUITTING;
	if (search_match)
	    break;
    }
    return WAITING_FOR_ARTICLE_REPLY;
}

/*
* Process the reply from an AUTHINFO USER command
* ENTRY: Notification has been received that data is ready to read
* This is executed once.
*/
int process_authinfo_user_reply(stateMachine s)
{
    int st;

    if ((st = get_one_line(szResponse)) < 0)
	return QUITTING;

    bzero(reply, BUFSIZE);
    if (2 > (sscanf(szResponse, "%u %[^\r\n]", &status, reply))) {
	MessageBox(s.hwndDlg,
	    "Can't read AUTHINFO USER reply",
	    "Message parse error", MB_OK);
	return QUITTING;
    }
    if (status == MORE_AUTH_REQUIRED) {
	bzero(sendbuf, BUFSIZE);
	sprintf(sendbuf, "AUTHINFO PASS %s\r\n", s.szPassword);
	st = send(s.sockNews, sendbuf, strlen(sendbuf), 0);
	return WAITING_FOR_AUTHINFO_PASS_REPLY;
    }
    if (status == AUTH_OK) {
	if (s.fListNewsgroups) {
	    bzero(sendbuf, BUFSIZE);
	    sprintf(sendbuf, "LIST NEWSGROUPS\r\n");
	    st = send(s.sockNews, sendbuf, strlen(sendbuf), 0);
	    return WAITING_FOR_LIST_NEWSGROUPS_REPLY;
	} else {
	    bzero(sendbuf, BUFSIZE);
	    sprintf(sendbuf, "GROUP %s\r\n", s.szNewsgroup);
	    st = send(s.sockNews, sendbuf, strlen(sendbuf), 0);
	    return WAITING_FOR_GROUP_REPLY;
	}
    }
    if (status == AUTH_FAILED) {
	MessageBox(s.hwndDlg,
	    "Login failed",
	    "Authorization Error", MB_OK | MB_ICONEXCLAMATION);
	return AUTHORIZATION_FAILED;
    }
    sprintf(gszTemp,
	"Strange login status: %d\r\nPlease email crispen@hiwaay.net",
	status);
    MessageBox(s.hwndDlg,
	gszTemp,
	"Authorization Error", MB_OK | MB_ICONSTOP);
    return AUTHORIZATION_FAILED;
}

/*
* Process the reply from an AUTHINFO PASS command
* ENTRY: Notification has been received that data is ready to read
* This is executed once.
*/
int process_authinfo_pass_reply(stateMachine s)
{
    int st;

    if ((st = get_one_line(szResponse)) < 0)
	return QUITTING;

    bzero(reply, BUFSIZE);
    if (2 > (sscanf(szResponse, "%u %[^\r\n]", &status, reply))) {
	MessageBox(s.hwndDlg,
	    "Can't read AUTHINFO PASS reply",
	    "Message parse error", MB_OK);
	return QUITTING;
    }
    if (status == MORE_AUTH_REQUIRED) {
	MessageBox(s.hwndDlg,
	    "Server requires more authorization\r\nPlease email crispen@hiwaay.net",
	    "Authorization Error", MB_OK | MB_ICONSTOP);
	return AUTHORIZATION_FAILED;
    }
    if (status == AUTH_OK) {
	if (s.fListNewsgroups) {
	    bzero(sendbuf, BUFSIZE);
	    sprintf(sendbuf, "LIST NEWSGROUPS\r\n");
	    st = send(s.sockNews, sendbuf, strlen(sendbuf), 0);
	    return WAITING_FOR_LIST_NEWSGROUPS_REPLY;
	} else {
	    bzero(sendbuf, BUFSIZE);
	    sprintf(sendbuf, "GROUP %s\r\n", s.szNewsgroup);
	    st = send(s.sockNews, sendbuf, strlen(sendbuf), 0);
	    return WAITING_FOR_GROUP_REPLY;
	}
    }
    if (status == AUTH_FAILED) {
	MessageBox(s.hwndDlg,
	    "Login failed",
	    "Authorization Error", MB_OK | MB_ICONEXCLAMATION);
	return AUTHORIZATION_FAILED;
    }
    sprintf(gszTemp,
	"Strange password status: %d\r\nPlease email crispen@hiwaay.net",
	status);
    MessageBox(s.hwndDlg,
	gszTemp,
	"Authorization Error", MB_OK | MB_ICONSTOP);
    return AUTHORIZATION_FAILED;
}

/*
* Clean up and quit
*/
int process_quitting(stateMachine s)
{
    int i;

    // Clear the progress window
    SendMessage(s.hwndProgress, PBM_SETPOS, (WPARAM)0, 0L);

    // Clear the highlighted text in the headers window
    SendMessage(hwndEdit2, EM_SETSEL, 0, 0);

    // Close any files that are open
    if (fd)
	fclose(fd);
    if (afd)
	fclose(afd);
    if (arfd)
	fclose(arfd);
    if (urlfd) {
	fprintf(urlfd, "</UL>\n");
	fclose(urlfd);
    }
    if (sfd)
	fclose(sfd);
    if (logfd)
	fclose(logfd);

    Display3("Closing connection to news server");
    sprintf(gszTemp, "QUIT\r\n");
    (void)send(s.sockNews, gszTemp, strlen(gszTemp), 0);

    // Drain off any extra messages coming from the news host
    while (0< (recv(s.sockNews, rawbuf, ETHER_MTU, 0)));
    if (nMatches > 0) {
	sprintf(gszTemp,
	   "Saved %d attached files from %d articles matching search string",
	   nAttachments, nMatches);
	Display(gszTemp);
    } else {
	Display("Connection Closed");
    }

    // Free any memory allocated
    if (newsgroup_list_count !=0) {
	for (i=0; i<newsgroup_list_count; i++) {
	    free(newsgroup_list[i]);
	}
	free(newsgroup_list);
    }

    // Reinitialize variables
    nMatches = 0;
    nAttachments = 0;
    init_buffers();

    shutdown(s.sockNews, 2);
    closesocket(s.sockNews);
    quit_signal_rcvd = FALSE;
    return QUITTING;
}

/************************ OTHER VISIBLE FUNCTIONS *************************/

/*
* Set the quit flag
*/
void signal_quit(BOOL f)
{
    quit_signal_rcvd = f;
}

/*************************** AUXILIARY ROUTINES ***************************/

/*
* Initialize the buffers and pointers
*/
void init_buffers()
{
    bzero(rawbuf, BUFSIZE);
    bzero(current_line, BUFSIZE);
    rawbuf_ptr = 0;
    current_line_ptr = 0;
    bytes_rcvd = 0;
}

/*
* Receive a buffer from the news host
* RETURN: FALSE if OK, TRUE if error
*/
BOOL recv_from_news_host(HWND hwndDlg, SOCKET s)
{
    bzero(rawbuf, BUFSIZE);
    bytes_rcvd = recv(s, rawbuf, ETHER_MTU, 0);
    if (bytes_rcvd == SOCKET_ERROR) {
	MessageBox(hwndDlg, "Socket read error in recv_from_news",
	"recv() error", MB_OK);
	return TRUE;
    }
    if (bytes_rcvd == 0) {
	MessageBox(hwndDlg,
	    "Connection closed unexpectedly",
	    "recv() error", MB_OK);
	return TRUE;
    }
    rawbuf_ptr = 0;
    return FALSE;
}

/*
* Get one line from the buffer.
* RETURN:
*  Size of line if OK
*  -1 if end of section[1]
*   0 if partial buffer remaining or empty buffer
*
* [1] Per RFC 977 all lines to and from the news host are terminated by \r\n
* End of section is a line consisting of ".\r\n"
*/
int get_one_line(char *buf)
{
    int i, j;

    bzero(buf, BUFSIZE);
    /*
    * Don't wipe out the current line if there's something in it
    */
    bzero(&current_line[current_line_ptr], BUFSIZE-current_line_ptr);
    for (i=rawbuf_ptr; i<bytes_rcvd; i++) {
	current_line[current_line_ptr++] = rawbuf[i];
	if (rawbuf[i] == '\n') {
	    /*
	    * See if this line is the last line of the section
	    */
	    if (!strcmp(current_line, ".\r\n")) {
		strcpy(buf, current_line);
		init_buffers();
		return -1;
	    }
	    /*
	    * Must be the end of a line from the section.  Return the
	    * current line.
	    */
	    j = current_line_ptr;
	    strcpy(buf, current_line);
	    current_line_ptr = 0;
	    rawbuf_ptr = i+1;
	    return j;
	}
    }
    /*
    * If we end up here, either the line is incomplete or the buffer
    * is empty.
    */
    rawbuf_ptr = 0;
    return 0;
}

/*
* Increment the header count.  If we haven't reached the last one yet,
* send another HEAD command.  If we have, then depending on what we're
* supposed to do next, we either set up to read article, or we list
* the headers.
*/
int next_header(stateMachine s)
{
    int st;

    if (++article_num > last) {
	if (s.fShowHdrs) {
	    list_headers(
		s.hwndDlg, s.hwndProgress, s.fPastDays, s.szSearchString);
	    return QUITTING;
	} else {
	    if ((set_up_for_reading_articles(s)) == -1) {
		return QUITTING;
	    }
	    return WAITING_FOR_ARTICLE_REPLY;
	}
    }
    sprintf(sendbuf, "HEAD %u\r\n", article_num);
    st = send(s.sockNews, sendbuf, strlen(sendbuf), 0);
    return WAITING_FOR_HEAD_REPLY;
}

/*
* List the header lines from the file that match the date criterion
* This is executed once
*/
void list_headers(
    HWND hwndDlg, HWND hwndProgress, BOOL pastDays, char* szSearchString)
{
    char* sz;
    int disp;
    unsigned long int lFirst, lLast, lNumArticles;
    time_t thisDate;
    struct stat ss;
    int lDay;
    int i;

    Display3("This may take a while if you've got a lot of headers");
    // Clear out the progress bar
    SendMessage(hwndProgress, PBM_SETPOS, (WPARAM)0, 0L);

    // Clear out the header display window
    SendMessage(hwndEdit2, EM_SETSEL, 0, -1);
    SendMessage(hwndEdit2, WM_CLEAR, 0, 0);
    SendMessage(hwndEdit2, EM_SETSEL, 0, 0);

    // Header line file was left open for writing.  Close it and
    // open it for reading.
    fclose(fd);
    stat(filename, &ss);
    SendMessage(hwndEdit2, EM_SETLIMITTEXT, (WPARAM)ss.st_size, 0);

    fd = fopen(filename, "rb");
    if (fd == 0) {
	MessageBox(hwndDlg,
	    "Can't open header file for read",
	    "fopen() error", MB_OK);
	return;
    }
    hdr_file_line = 1;

    // Better be able to get the first line of that file!
    bzero(subj, BUFSIZE);
    sz = fgets(subj, BUFSIZE, fd);
    if (sz == NULL) {
	MessageBox(hwndDlg,
	    "Can't get first line of header file",
	    "Message parse error", MB_OK);
	return;
    }
    if (3 > (sscanf(subj, "%u %u %u", &lNumArticles, &lFirst, &lLast))) {
	MessageBox(hwndDlg,
	    "Can't parse first line of header file",
	    "Message parse error", MB_OK);
	return;
    }

    bzero(readingHeadersLine, BUFSIZE);
    if (pastDays) {
	sprintf(readingHeadersLine,"Displaying headers");
    } else {
	sprintf(readingHeadersLine,"Displaying %u newsgroup headers",
	    lNumArticles);
    }
    Display(readingHeadersLine);

    for(;;) {
	if ((an = get_hdr_file_line(hwndDlg, szSearchString)) == -1)
	    return;
	if (lLast != lFirst) {
	    disp = (100*(an-lFirst))/(lLast-lFirst);
	    if (disp > 100)
		disp = 100;
	    SendMessage(hwndProgress, PBM_SETPOS, (WPARAM)disp, 0L);
	    sprintf(gszTemp, "%s %d%%", readingHeadersLine, disp);
	    Display(gszTemp);
	}

	// Put 2 spaces in front of single digit dates to make our display nicer
	sscanf(szMsgDate, "%d", &lDay);
	if (lDay < 10) {
	    for (i=strlen(szMsgDate)+1; i>0; --i)
		szMsgDate[i] = szMsgDate[i-2];
	    szMsgDate[0] = ' ';
	    szMsgDate[1] = ' ';
	}

	if (pastDays) {
	    // Convert the message data to a time_t
	    thisDate = date_to_time_t(szMsgDate, hwndDlg);
	    if (thisDate == -1)
		return;
	    if (thisDate >= dateLimit) {
		sprintf(gszTemp, "%u\t%s\t%s\r\n",
		    nLines, szMsgDate, szSubject);
		Display2(gszTemp);
	    }
	} else {
	    sprintf(gszTemp, "%u\t%s\t%s\r\n",
		nLines, szMsgDate, szSubject);
	    Display2(gszTemp);
	}
    }
}

/*
* Set up for reading article.
*  Close the header file if it's open.
*  Reopen the header file in read mode and get the first article number.
*  Send an ARTICLE command to the news server for the first article.
* This should only execute once.
*/
int set_up_for_reading_articles(stateMachine s)
{
    char* sz;
    unsigned long int lFirst, lLast, lNumArticles;

    SendMessage(s.hwndProgress, PBM_SETPOS, (WPARAM)0, 0L);
    Display3("Reading articles");

    nMatches = 0;
    nAttachments = 0;

    hdr_file_line = 1;
    fclose(fd);
    fd = fopen(filename, "rb");
    if (fd == 0) {
	MessageBox(s.hwndDlg,
	    "Can't open header file for read",
	    "fopen() error", MB_OK);
	return -1;
    }
    init_buffers();
    bzero(subj, BUFSIZE);
    sz = fgets(subj, BUFSIZE, fd);
    if (sz == NULL)
	return -1;
    if (3 > (sscanf(subj, "%u %u %u",
	&lNumArticles, &lFirst, &lLast))) {
	MessageBox(s.hwndDlg,
	    "Can't parse first line of header file",
	    "Message parse error", MB_OK);
	return -1;
    }
    sprintf(gszTemp, "Searching for articles that match \"%s\"",
	s.szSearchString);
    Display(gszTemp);
    for(;;) {
	an = get_hdr_file_line_and_send_body(
	    s.hwndDlg, s.hwndProgress, s.sockNews, s.szSearchString);
	if (an == -1) {
	    return -1;
	}
	if (search_match) {
	    break;
	}
    }

    // Open the URL file if the user has asked to save URLs
    if (s.fURLs) {
	if ((open_url_file(s.hwndDlg, s.szNewsgroup)) == -1) {
	    return QUITTING;
	}
    }

    findPtr = 0;
    return 0;
}

/*
* Re-create the header file, copying only articles that are within the
* article limits on the server
*/
int copy_header_file(HWND hwndDlg) {

    char *sz;
    int skipped_last;

    fclose(fd);
    bzero(scratchfilename, BUFSIZE);
    strcpy(scratchfilename, filename);
    scratchfilename[strlen(scratchfilename)-1] = 'x';
    if ((rename(filename, scratchfilename)) < 0) {
	MessageBox(hwndDlg,
	    "Can't rename header file",
	    "rename() error", MB_OK);
	return -1;
    }
    sfd = fopen(scratchfilename, "rb");
    if (sfd == 0) {
	MessageBox(hwndDlg,
	    "Can't open scratch file for read",
	    "fopen() error", MB_OK);
	return -1;
    }
    fd = fopen(filename, "wb");
    if (fd == 0) {
	MessageBox(hwndDlg,
	    "Can't create header file",
	    "fopen() error", MB_OK);
	return -1;
    }
    fprintf(fd, "%u %u %u\r\n", num_articles, first, last);
    sz = fgets(subj, BUFSIZE, sfd);
    if (sz == NULL) {
	MessageBox(hwndDlg,
	    "Can't read first line of scratch file",
	    "fgets() error", MB_OK);
	return -1;
    }
    /*
    * Copy only the articles that are still on the server
    */
    skipped_last = 0;
    bzero(subj, BUFSIZE);
    while ((sz = fgets(subj, BUFSIZE, sfd))) {
	if (isspace(subj[0])) {
	    if (!skipped_last) {
		fprintf(fd, "%s", subj);
	    }
	} else {
	    if (2 > (sscanf(subj, "%u %u ", &an, &nLines))) {
		MessageBox(hwndDlg,
		    "Can't get article number in copy header file",
		    "Message parse error", MB_OK);
		/*
		* The header file is probably garbage.  Get rid of it
		* and the scratch file and maybe we'll have better luck
		* next time we have to build it.
		*/
		fclose(sfd);
		fclose(fd);
		(void)unlink(scratchfilename);
		(void)unlink(filename);
		return -1;
	    }
	    /*
	    * Copy only the articles between first and last
	    */
	    if ((an >= first) && (an <= last)) {
		fprintf(fd, "%s", subj);
		skipped_last = 0;
	    } else {
		skipped_last = 1;
	    }
	}
	bzero(subj, BUFSIZE);
    }
    fclose(fd);
    fclose(sfd);
    if ((unlink(scratchfilename)) < 0) {
	MessageBox(hwndDlg,
	    "Can't unlink scratch file",
	    "unlink() error", MB_OK);
	return -1;
    }
    return 0;
}

/*
* Open a file called <newshost>\Attachments\<newsgroup>\links.html
* to which we'll copy any URLs found in the article.
*/
int open_url_file (HWND hwndDlg, char* szNewsgroup)
{
    int j;
    struct stat ss;
    char urlfn[BUFSIZE];

    // Start off the URL file name with the news host name
    bzero(urlfn, BUFSIZE);
    strcpy(urlfn, filename);
    for (j=strlen(urlfn)-1; urlfn[j] != '\\'; --j)
	urlfn[j] = '\0';

    // Create the Attachments directory if it doesn't exist yet
    strcat(urlfn, "Attachments");
    if ((stat(urlfn, &ss)) != 0) {
	if (!CreateDirectory(urlfn, NULL)) {
	    sprintf(gszTemp, "Can't create folder %s", urlfn);
	    MessageBox(hwndDlg, gszTemp, "CreateDirectory() error", MB_OK);
	    return -1;
	}
    } else if (!(ss.st_mode & S_IFDIR)) {
	sprintf(gszTemp, "File named %s exists, but isn't a folder", urlfn);
	MessageBox(hwndDlg, gszTemp, "stat() error", MB_OK);
	return -1;
    }

    // Create the newsgroup directory if it doesn't exist yet
    urlfn[strlen(urlfn)] = '\\';
    strcat(urlfn, szNewsgroup);
    if ((stat(urlfn, &ss)) != 0) {
	if (!CreateDirectory(urlfn, NULL)) {
	    sprintf(gszTemp, "Can't create folder %s", urlfn);
	    MessageBox(hwndDlg, gszTemp, "CreateDirectory() error", MB_OK);
	    return -1;
	}
    } else if (!(ss.st_mode & S_IFDIR)) {
	sprintf(gszTemp, "File named %s exists, but isn't a folder", urlfn);
	MessageBox(hwndDlg, gszTemp, "fopen() error", MB_OK);
	return -1;
    }

    // Create the URL file
    strcat(urlfn, "\\links.html");
    if ((stat(urlfn, &ss)) == -1) {
	if (errno == ENOENT) {
	    urlfd = fopen(urlfn, "w");
	    fprintf(urlfd,
		"<CENTER>\n<H2>URLs found in %s</H2>\n</CENTER>\n<P>\n<UL>\n",
		szNewsgroup);
	} else {
	    MessageBox(hwndDlg,
		"Can't open URL file",
		"stat() error", MB_OK);
	    return -1;
	}
    } else {
	urlfd = fopen(urlfn, "a");
	fprintf(urlfd, "<P>\n<UL>\n");
    }
    first_url_found = FALSE;
    return 0;
}

/*
* Open a file called <newshost>\Articles\<newsgroup>\Article_xxx
* to which we'll copy the attachments.
*/
int open_article_file(HWND hwndDlg, char* szNewsgroup)
{
    char afn[BUFSIZE];
    int j;
    struct stat ss;

    bzero(afn, BUFSIZE);
    strcpy(afn, filename);
    for (j=strlen(afn)-1; afn[j] != '\\'; --j)
	afn[j] = '\0';

    // Create the Articles directory if it doesn't exist yet
    strcat(afn, "Articles");
    if ((stat(afn, &ss)) != 0) {
	if (!CreateDirectory(afn, NULL)) {
	    sprintf(gszTemp, "Can't create folder %s", afn);
	    MessageBox(hwndDlg, gszTemp, "CreateDirectory() error", MB_OK);
	    return -1;
	}
    } else if (!(ss.st_mode & S_IFDIR)) {
	sprintf(gszTemp, "File named %s exists, but isn't a folder", afn);
	MessageBox(hwndDlg, gszTemp, "stat() error", MB_OK);
	return -1;
    }

    // Create the <newsgroup> directory if it doesn't exist yet
    afn[strlen(afn)] = '\\';
    strcat(afn, szNewsgroup);
    if ((stat(afn, &ss)) != 0) {
	if (!CreateDirectory(afn, NULL)) {
	    sprintf(gszTemp, "Can't create folder %s", afn);
	    MessageBox(hwndDlg, gszTemp, "CreateDirectory() error", MB_OK);
	    return -1;
	}
    } else if (!(ss.st_mode & S_IFDIR)) {
	sprintf(gszTemp, "File named %s exists, but isn't a folder", afn);
	MessageBox(hwndDlg, gszTemp, "fopen() error", MB_OK);
	return -1;
    }

    // Create the article file
    afn[strlen(afn)] = '\\';
    sprintf(gszTemp, "Article_%3.3u.txt", ++article_count);
    strcat(afn, gszTemp);
    if ((create_article_file(hwndDlg, afn)) == -1) {
	sprintf(gszTemp, "Can't create %s", afn);
	MessageBox(hwndDlg, gszTemp, "fopen() error", MB_OK);
	return -1;
    }
    return 0;
}

/*
* Open a file called <newshost>\Attachments\<newsgroup>\<filename>
* If the file or directory don't exist, create them.
* If the file does exist, create a file with a new filename.
* We can assume that the filename has already been cleaned up.
*/
int open_attachment_file(
    HWND hwndDlg, char* szNewsgroup, char* afn, BOOL fMakeLog,
    BOOL fDontSaveDups)
{
    int j;
    struct stat ss;

    // Start off with the news host name
    bzero(attfn, BUFSIZE);
    strcpy(attfn, filename);
    for (j=strlen(attfn)-1; attfn[j] != '\\'; --j)
	attfn[j] = '\0';

    // Create the Attachments directory if it doesn't exist yet
    strcat(attfn, "Attachments");
    if ((stat(attfn, &ss)) != 0) {
	if (!CreateDirectory(attfn, NULL)) {
	    sprintf(gszTemp, "Can't create folder %s", attfn);
	    MessageBox(hwndDlg, gszTemp, "CreateDirectory() error", MB_OK);
	    return -1;
	}
    } else if (!(ss.st_mode & S_IFDIR)) {
	sprintf(gszTemp, "File named %s exists, but isn't a folder", attfn);
	MessageBox(hwndDlg, gszTemp, "stat() error", MB_OK);
	return -1;
    }

    // Create the newsgroup directory if it doesn't exist yet
    attfn[strlen(attfn)] = '\\';
    strcat(attfn, szNewsgroup);
    if ((stat(attfn, &ss)) != 0) {
	if (!CreateDirectory(attfn, NULL)) {
	    sprintf(gszTemp, "Can't create folder %s", attfn);
	    MessageBox(hwndDlg, gszTemp, "CreateDirectory() error", MB_OK);
	    return -1;
	}
    } else if (!(ss.st_mode & S_IFDIR)) {
	sprintf(gszTemp, "File named %s exists, but isn't a folder", attfn);
	MessageBox(hwndDlg, gszTemp, "fopen() error", MB_OK);
	return -1;
    }

    // Set the name for the log file if the user asks for one
    if (fMakeLog) {
	// Make a filename for the log file
	bzero(logfn, BUFSIZE);
	strcpy(logfn, attfn);
	strcat(logfn, "\\log.txt");
    }

    // Create the attachment file
    attfn[strlen(attfn)] = '\\';
    strcat(attfn, afn);
    return (create_attachment_file(
	hwndDlg, attfn, fMakeLog, fDontSaveDups));
}

/*
* Create a new file for the attachment
*/
int create_attachment_file(HWND hwndDlg, char* afn, BOOL fMakeLog,
    BOOL fDontSaveDups)
{
    struct stat ss;
    int i, j, sl;
    char oldfilename[BUFSIZE];
    char fnextra[BUFSIZE];
    char extension[BUFSIZE];
    char* basename;
    SYSTEMTIME sTime;
#define TRIES 256

    if ((stat(afn, &ss)) == -1) {
	if (errno == ENOENT) {
	    afd = fopen(afn, "wb");
	} else {
	    MessageBox(hwndDlg,
		"Can't open attachments file",
		"stat() error", MB_OK);
	    return -1;
	}
    } else if (fDontSaveDups) {
	basename = strrchr(afn, '\\');
	sprintf(gszTemp, "Not saving %s - file exists",
	    (basename ? &basename[1] : afn));
	Display(gszTemp);
	return 0;
    } else {
	bzero(oldfilename, BUFSIZE);
	strcpy(oldfilename, afn);
	sl = strlen(oldfilename);
	for (i=sl-1; ((i > 0) && (oldfilename[i] != '.')); --i);
	bzero(extension, BUFSIZE);
	if (i>0) {
	    j=0;
	    for (; i<sl; i++) {
		extension[j++] = oldfilename[i];
		oldfilename[i] = '\0';
	    }
	}
	for (i=1; i<TRIES; i++) {
	    bzero(afn, BUFSIZE);
	    strcpy(afn, oldfilename);
	    bzero(fnextra, BUFSIZE);
	    sprintf(fnextra, ".%u", i);
	    strcat(afn, fnextra);
	    strcat(afn, extension);
	    if ((stat(afn, &ss)) == -1) {
		if (errno == ENOENT) {
		    afd = fopen(afn, "wb");
		    break;
		}
	    }
	}
	if (i >= TRIES) {
	    MessageBox(hwndDlg,
		"More than 255 duplicate filenames",
		"stat() error", MB_OK);
	    return -1;
	}
    }
    if (afd == 0) {
	sprintf(gszTemp, "Can't create %s", afn);
	MessageBox(hwndDlg,
	    gszTemp,
	    "fopen() error", MB_OK);
	return -1;
    }
    basename = strrchr(afn, '\\');
    bzero(savingFileName, BUFSIZE);
    sprintf(savingFileName, "Saving %s", (basename ? &basename[1] : afn));
    Display(savingFileName);

    // Write the sender, date, and subject of the file into a log file
    if (fMakeLog) {
	if ((stat(logfn, &ss)) == -1) {
	    if (errno == ENOENT) {
		logfd = fopen(logfn, "w");
		fprintf(logfd,
"WARNING: SCAN THESE FILES FOR VIRUSES BEFORE OPENING ANY OF THEM!\n");
		fprintf(logfd,
"-----------------------------------------------------------------\n\n");
	    } else {
		MessageBox(hwndDlg,
		    "Can't open log file",
		    "stat() error", MB_OK);
		return -1;
	    }
	} else {
	    logfd = fopen(logfn, "a");
	}
	fprintf(logfd, "From: %s\n", szSender);
	fprintf(logfd, "Date: %s\n", szMsgDate);
	fprintf(logfd, "Subject: %s\n", szSubject);
	fprintf(logfd, "File: %s\n", (basename ? &basename[1] : afn));
	GetLocalTime(&sTime);
	strncpy(gszTemp, monthNames[sTime.wMonth-1], 3);
	gszTemp[3] = '\0';
	strcpy(&gszTemp[4], "AM");
	gszTemp[6] = '\0';
	if (sTime.wHour > 12) {
	    sTime.wHour -=12;
	    gszTemp[4] = 'P';
	}
	fprintf(logfd, "Downloaded: %d %s %d, %d:%d %s\n\n",
	    sTime.wDay, gszTemp, sTime.wYear, sTime.wHour, sTime.wMinute,
	    &gszTemp[4]);
	fclose(logfd);
    }

    extracting_attachment = TRUE;
    nAttachments++;
    return 0;
}

/*
* Create a file for the article
*/
int create_article_file(HWND hwndDlg, char* afn)
{
    struct stat ss;
    int i, j, sl;
    char oldfilename[BUFSIZE];
    char fnextra[BUFSIZE];
    char extension[BUFSIZE];
    char* basename;
#define TRIES 256

    if ((stat(afn, &ss)) == -1) {
	if (errno == ENOENT) {
	    arfd = fopen(afn, "wb");
	} else {
	    MessageBox(hwndDlg,
		"Can't open attachments file",
		"stat() error", MB_OK);
	    return -1;
	}
    } else {
	bzero(oldfilename, BUFSIZE);
	strcpy(oldfilename, afn);
	sl = strlen(oldfilename);
	for (i=sl-1; ((i > 0) && (oldfilename[i] != '.')); --i);
	bzero(extension, BUFSIZE);
	if (i>0) {
	    j=0;
	    for (; i<sl; i++) {
		extension[j++] = oldfilename[i];
		oldfilename[i] = '\0';
	    }
	}
	for (i=1; i<TRIES; i++) {
	    bzero(afn, BUFSIZE);
	    strcpy(afn, oldfilename);
	    bzero(fnextra, BUFSIZE);
	    sprintf(fnextra, "(%u)", i);
	    strcat(afn, fnextra);
	    strcat(afn, extension);
	    if ((stat(afn, &ss)) == -1) {
		if (errno == ENOENT) {
		    arfd = fopen(afn, "wb");
		    break;
		}
	    }
	}
	if (i >= TRIES) {
	    MessageBox(hwndDlg,
		"More than 255 duplicate filenames",
		"stat() error", MB_OK);
	    return -1;
	}
    }
    if (arfd == 0) {
	sprintf(gszTemp, "Can't create %s", afn);
	MessageBox(hwndDlg,
	    gszTemp,
	    "fopen() error", MB_OK);
	return -1;
    }
    basename = strrchr(afn, '\\');
    bzero(savingFileName, BUFSIZE);
    sprintf(savingFileName, "Saving %s", (basename ? &basename[1] : afn));
    Display(savingFileName);
    extracting_article = TRUE;
    return 0;
}


/*
* Get the next line from the header file.  If the subject line matches
* our search string, issue an ARTICLE command to the server to fetch the
* body of the corresponding article.
*/
int get_hdr_file_line_and_send_body(
    HWND hwndDlg, HWND hwndProgress, SOCKET sockNews, char* szSearchString)
{
    int st;

    in_hdr = TRUE;
    extracting_attachment = FALSE;
    extracting_article = FALSE;
    uudecoding = FALSE;
    this_line = 0;
    search_match = FALSE;

    SendMessage(hwndProgress, PBM_SETPOS, (WPARAM)0, 0L);
    an = get_hdr_file_line(hwndDlg, szSearchString);
    if (an == -1)
	return -1;
    if (search_match) {
	nMatches++;
	sprintf(sendbuf, "ARTICLE %u\r\n", an);
	st = send(sockNews, sendbuf, strlen(sendbuf), 0);
    }
    return an;
}

/*
* Get the next line from the header file.
* RETURN: article number or -1 for error.
*/
int get_hdr_file_line(HWND hwndDlg, char* szSearchString)
{
    char* sz;
    char uglysubject[BUFSIZE];
    time_t thisDate;

    bzero(subj, BUFSIZE);
    sz = fgets(subj, BUFSIZE, fd);
    if (sz == NULL)
	return -1;

    hdr_file_line++;
    // Get the fields from the line
    bzero(szSubject, BUFSIZE);
    if (6 > (sscanf(subj, "%u %u %u %[^\t] %[^\t] %[^\r\n]",
	&an, &nLines, &nBytes, szMsgDate, szSender, szSubject))) {
	sprintf(gszTemp, "Can't read line %d in %s", hdr_file_line, filename);
	MessageBox(hwndDlg,
	    gszTemp,
	    "Message parse error", MB_OK);
	return -1;
    }

    // If the date wasn't on the list we displayed, no match
    thisDate = date_to_time_t(szMsgDate, hwndDlg);
    if (thisDate == -1)
	return -1;
    if (thisDate < dateLimit)
	return an;

    // See if we have a match with our search string
    bzero(uglysubject, BUFSIZE);
    for (int i=0; i<strlen(szSubject); i++) {
	if (islower(szSubject[i]))
	    uglysubject[i] = toupper(szSubject[i]);
	else
	    uglysubject[i] = szSubject[i];
    }

    search_match = match(szSearchString, uglysubject);
    return an;
}

/*
* Pattern match with wildcards.  Thanks to Joan Ordinas.
*/
int match(char* pattern, char* string)
{
    enum {  false, true,
	ANY = '?', ZERO_OR_MORE = '*',
	START_CLASS = '[', END_CLASS = ']'
	};

    if (pattern[0] == '\0' && string[0] == '\0') {
	return true;
    } else if (pattern[0] == '\0' || string[0] == '\0') {
	return false;
    } else if (pattern[0] == ANY || pattern[0] == string[0]) {
	return match(pattern + 1, string + 1);
    } else if (pattern[0] == ZERO_OR_MORE) {
	return  pattern[1] == '\0' ||
	    match(pattern + 1, string) ||
	    match(pattern, string + 1);
    } else if (pattern[0] == START_CLASS) {
	const char *end;
	if ((end = strchr(pattern, END_CLASS)) == NULL) {
	    return false;   
	} else {
	    int n = -1;
	    char class[32] = "%*1";

	    strncat(class+3, pattern, end - pattern + 1);
	    if (class[4] == '!') { class[4] = '^'; }
	    strcat(class, "%n");
	    sscanf(string, class, &n);
	    return (n == 1) && match(end + 1, string + 1);
	}
    } /* else */
	return false;
}

/*
* Uudecode
*/
void uudecode(char* buf, FILE* fd)
{
#define DEC(c)	(((c) - ' ') & 077)
    int n;
    char *bp;

    if (!strncmp(buf, "begin", 5))
	return;
    if (!strncmp(buf, "end", 3))
	return;
    n = DEC(buf[0]);
    if (n <= 0)
	return;
    bp = &buf[1];
    while (n > 0) {
	outdec(bp, fd, n);
	bp += 4;
	n -= 3;
    }
}
void outdec(char* p, FILE* f, int n)
{
    int c1, c2, c3;

    c1 = DEC(*p) << 2 | DEC(p[1]) >> 4;
    c2 = DEC(p[1]) << 4 | DEC(p[2]) >> 2;
    c3 = DEC(p[2]) << 6 | DEC(p[3]);
    if (n >= 1)
	putc(c1, f);
    if (n >= 2)
	putc(c2, f);
    if (n >= 3)
	putc(c3, f);
}

char b64tbl[66] =
    "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";
/*
* Decode a base64 attachment
*/
int b64decode(char* buf, FILE* fd, char* out)
{
    int i, j, k;
    int d, num, len, val;
    char nw[4];
    char *p;
    char *c;
    char newbuf[BUFSIZE];
    int decoded = 0;
    int outptr=0;

    bzero(newbuf, BUFSIZE);
    strcpy(newbuf, buf);
    if (newbuf[strlen(newbuf)-2] == '\r') {
	newbuf[strlen(newbuf)-2] = '\n';
	newbuf[strlen(newbuf)-1] = '\0';
    }
    // Enormous kludge -- if the file pointer is null, write to a string.
    // If the string pointer is also null, it means we haven't opened the
    // output file yet.
    if ((fd == NULL) && (out == NULL))
	return 0;
    if ((strchr(newbuf, ':')) || (strstr(newbuf, "name=")))
	return 0;
    len=strlen(newbuf)-1;
    for (i=0; i<len; i+=4) {
	val=0;
	num=3;
	c = newbuf+i;
	if (c[2] == '=')
	    num=1;
	else if (c[3] == '=')
	    num=2;

	for (j=0; j<=num; j++) {
	    p=strchr(b64tbl, c[j]);
	    d=p-b64tbl;
	    d<<=(3-j)*6;
	    val+=d;
	}
	for (j=2; j>=0; j--) {
	    nw[j]=val & 0xff;
	    val>>=8;
	}
	if (fd) {
	    fwrite(nw, 1, num, fd);
	} else {
	    for(k=0; k<num; k++) {
		// Get rid of tab, cr, lf, etc.
		if ((szResponse[0] == '\t') || (szResponse[0] == '\r') ||
		    (szResponse[0] == '\n') || (szResponse[0] == '\f') ||
		    (szResponse[0] == '\v') || (szResponse[0] == '\0')) {
		    out[outptr++] = ' ';
		} else {
		    out[outptr++] = nw[k];
		}
		decoded++;
	    }
	}
    }
    return(decoded);
}

/*
* Convert date string (01 JAN 1999) to a time_t
*/
time_t date_to_time_t(char *date, HWND hwndDlg)
{
    struct tm broken;
    time_t fixed;
    unsigned int day;
    char month[BUFSIZE];
    unsigned int year;
    int i;

    bzero(&broken, sizeof(struct tm));
    bzero(month, BUFSIZE);
    if (3 > (sscanf(date, "  %u %s %u", &day, month, &year))) {
	sprintf(gszTemp, "Can't convert \"%s\" to date", date);
	MessageBox(hwndDlg, gszTemp,
	"Date format error", MB_OK);
	return (time_t)-1;
    }
    for (i=0; i<12; i++) {
	if (!strncmp(month, monthNames[i], 3)) {
	    broken.tm_mon = i;
	    break;
	}
    }
    if (i==12) {
	sprintf(gszTemp, "Don't recognize month in %s", date);
	MessageBox(hwndDlg, gszTemp,
	"Date format error", MB_OK);
	return (time_t)-1;
    }
    broken.tm_mday = day;
    broken.tm_year = year - 1900;
    fixed = mktime(&broken);
    return fixed;
}

/*
* Convert a Windows SYSTEMTIME to a time_t
*/
time_t systime_to_time_t(SYSTEMTIME date)
{
    struct tm broken;
    time_t fixed;

    bzero(&broken, sizeof(struct tm));
    broken.tm_mday = date.wDay;
    broken.tm_year = date.wYear - 1900;
    broken.tm_mon = date.wMonth - 1;
    fixed = mktime(&broken);
    return fixed;
}

/*
* Return the earliest article date that we can accept headers from
*/
time_t time_limit(int days)
{
    SYSTEMTIME sTime;
    time_t offset;
    time_t today;

    GetLocalTime(&sTime);
    today = systime_to_time_t(sTime);
    offset = 60 * 60 * 24 * days;
    return today-offset;
}

/*
* Set the date limit
*/
void set_date_limit(daysPast)
{
    dateLimit = time_limit(daysPast);
}

/*
* Get the sender from the field pointed to on entry and store
* it in szSender.
*/
void get_sender(char* from_ptr, BOOL isTabSeparated, BOOL cat)
{
    get_hdr_text(from_ptr, szSender, isTabSeparated, cat);
}

/*
* Get the subject from the field pointed to on entry and store
* it in szSubject.
*/
void get_subject(char* subj_ptr, BOOL isTabSeparated, BOOL cat)
{
    get_hdr_text(subj_ptr, szSubject, isTabSeparated, cat);
}

/*
* Read in some header text.  If it isn't tab-separated, weed out
* any tabs that might be in it.  Convert any quoted-printable strings
* to equivalent text.  If the "cat" flag is set, concatenate it with
* a previous string.
*/
void get_hdr_text(char* in, char* out, BOOL isTabSeparated, BOOL cat)
{
    int i, j;
    char tText[BUFSIZE];

    bzero(tText, BUFSIZE);

    // Let the caller worry about having cleared out the destination buffer
    // if he asks for concatenation.
    if (!cat)
	bzero(out, BUFSIZE);

    // Clean out leading spaces and tabs
    for (i=0; ((in[i] == ' ') || (in[i] == '\t')); i++);

    if (isTabSeparated) {
	for (j=0; ((in[i] != '\t') && (in[i] != '\r')); j++) {
	    tText[j] = in[i++];
	}
    } else {
	// Detab the line
	// XXX - probably need to come back here and get rid of other
	// bogus characters later
	j = 0;
	for (; in[i] != '\r'; i++) {
	    if (in[i] == '\t')
		tText[j++] = ' ';
	    else
		tText[j++] = in[i];
	}
    }
    // See if we concatenate this onto the subject or just copy it
    if (cat)
	convert_alt_char_set(tText, &out[strlen(out)]);
    else
	convert_alt_char_set(tText, out);
}

/*
* Catch and convert any alternate character sets
* XXX - we assume that Windows can print the character.
*/
void convert_alt_char_set(char* in, char* out)
{

    int i, j, k, n;
    char szWork[BUFSIZE];
    char szWork2[BUFSIZE];
    char* sp;
    char* sp2;

    j = 0;

    // Get out quickly if we can
    sp = strstr(in, "=?");
    if (sp == NULL) {
	strcpy(out, in);
	return;
    }
    // We can output verbatim everything up to the questionable string
    for (sp2=in; sp2<sp; sp2++)
	out[j++] = *sp2;

    // Make a copy of the questionable part
    bzero(szWork, BUFSIZE);
    strcpy(szWork, sp);

    // Check for the designators we've been able to find so far
    if (!strnicmp(szWork, "=?iso-8859-1", 12)) {
	i = 12;
    } else if (!strnicmp(szWork, "=?iso-2022-", 11)) {
	i = 13;		// Country code is 2 characters long
    } else if (!strnicmp(szWork, "=?koi8-r", 8)) {
	i = 8;
    } else {
	// Beats me -- just copy it out and quit
	strcpy(out, in);
	return;
    }

    k = 0;
    // Quoted printable?
    if (!(strnicmp(&szWork[i], "?q?", 3))) {

	// Discard the designator
	for (i+=3; i<BUFSIZE; i++)
	    szWork[k++] = szWork[i];

	for (i=0; szWork[i] != '\0'; i++) {
	    if ((szWork[i] == '?') && (szWork[i+1] == '='))
		break;
	    if (szWork[i] == '=') {
		// XXX - assume the string is correct
		(void)sscanf(&szWork[i+1], "%x", &n);
		out[j++] = n;
		i += 2;
	    } else {
		out[j++] = szWork[i];
	    }
	}
	k=0;
	// Let's clean up again, just so we can debug this darn thing
	for (i+=2; i<BUFSIZE; i++)
	    szWork[k++] = szWork[i];
	if (szWork[0] != '\0')
	    // There might be yet another ISO string.  Let's try again.
	    convert_alt_char_set(szWork, &out[j]);

    // Binary (base64 encoded?)
    } else if (!(strnicmp(&szWork[i], "?b?", 3))) {

	// Discard the designator
	for (i+=3; i<BUFSIZE; i++)
	    szWork[k++] = szWork[i];
	// Now copy just the first encoded string to the buffer
	bzero(szWork2, BUFSIZE);
	for (i=0; szWork[i] != '\0'; i++) {
	    if ((szWork[i] == '?') && (szWork[i+1] == '='))
		break;
	    szWork2[i] = szWork[i];
	}
	k = 0;
	j+= b64decode(szWork2, (FILE*)NULL, &out[j]);
	for (i+=2; i<BUFSIZE; i++)
	    szWork[k++] = szWork[i];
	if (szWork[0] != '\0')
	    // There might be yet another ISO string.  Let's try again.
	    convert_alt_char_set(szWork, &out[j]);

    // Beats me what this string is, just copy it out
    } else {
	strcpy(out, in);
    }
}

/*
* The Date field in our file will consist of the numeric day, the month,
* and the year.  Ignore day of week, time, and time zone information.
*/
void get_date(char* date_ptr)
{
    int i, j;
    int lYear = 0;

    bzero(szMsgDate, BUFSIZE);
    szMsgDate[0] = '"';

    // Step ahead to the first digit.  Ignore leading zero.
    for (i=0; !isdigit(date_ptr[i]); i++);
    if (date_ptr[i] == '0')
	i++;
    j = 0;

    // Copy the day
    for (; isdigit(date_ptr[i]); i++)
	szMsgDate[j++] = date_ptr[i];

    // Copy the month
    for (; !isdigit(date_ptr[i]); i++)
	szMsgDate[j++] = date_ptr[i];

    // Copy the year
    sscanf(&date_ptr[i], "%d ", &lYear);
    if (lYear < 1970)
	lYear += 1900;
    if (lYear < 1970)
	lYear += 100;
    sprintf(&szMsgDate[j], "%d", lYear);
}

/*
* Clean up the file name to something the Windows filesystem might like
*/
void clean_up_filename(char* afn)
{
    char* sp;
    int i;
    char szFileName[BUFSIZE];

    bzero(szFileName, BUFSIZE);

    // Get rid of the full pathnames some moronic news agents put in
    sp = strrchr(afn, '\\');
    if (!sp) {
	sp = strrchr(afn, '/');
	if (!sp) {
	    sp = afn;
	} else {
	    ++sp;
	}
    } else {
	++sp;
    }
    if (strlen(sp) == 0) {
	bzero(afn, BUFSIZE);
	strcpy(afn, "something.bin");
    }
    sp[strlen(sp)] = '\t';

    // Some filenames contain MIME encoding
    get_hdr_text(sp, szFileName, TRUE, FALSE);

    // Now replace forbidden characters with underscores
    for (i=0; szFileName[i] != '\0'; i++) {
	if ((szFileName[i] == '|') || (szFileName[i] == '/') ||
	    (szFileName[i] == ':') || (szFileName[i] == '*') ||
	    (szFileName[i] == '?') || (szFileName[i] == '"') ||
	    (szFileName[i] == '<') || (szFileName[i] == '>') ||
	    (szFileName[i] == '\\'))
	    szFileName[i] = '_';
    }

    // If the filename doesn't have an extension, give it the
    // extension ".bin"
    if (!(strchr(szFileName, '.'))) {
	strcat(szFileName, ".bin");
    }

    bzero(afn, BUFSIZE);
    strcpy(afn, szFileName);
}

/*
* Compare strings function for qsort
*/
int compare_newsgroups (char**a, char** b)
{
    int i;
    char aa[BUFSIZE];
    char bb[BUFSIZE];

    bzero(aa, BUFSIZE);
    bzero(bb, BUFSIZE);
    for (i=0; i<strlen(a[0]) && a[0][i] != '\t'; i++)
	aa[i] = a[0][i];
    for (i=0; i<strlen(b[0]) && b[0][i] != '\t'; i++)
	bb[i] = b[0][i];
    return(strcmp(aa,bb));
}
