------------------------------------------------------------------------------
-- Unit Name:   Abbreviations
-- Unit Type:   package body
-- Source File: abbreviations.adb
-- Author:      Bob Crispen <crispen@hiwaay.net>
-- Date:        19 April 1996
-- Implementation Notes:
-- External Unit Declarations:
--
with Ada.Characters.Handling;
with Ada.Strings.Maps.Constants;
with Ada.Strings.Fixed;
--
-- Visibility Declarations: None
-- Unit Declaration:
--
package body Abbreviations is
--
-- Exceptions Raised: None
--
-- This software is provided under the terms of the Gnu General Public
-- License
------------------------------------------------------------------------------

   type String_Access is access String;

   ---------------------------------------------------------------------------
   -- The following abbreviations have been approved by program management
   -- for use in code produced for the program.  Note: abbreviations
   -- whose first letter is capitalized and all the rest of whose letters
   -- are in lower case need not be put on this list.
   ---------------------------------------------------------------------------
   Program_Abbreviations : constant array (1..6) of String_Access := (
      new String'("IO"),
      new String'("dB"),
      new String'("kVA"),
      new String'("CR"),
      new String'("LF"),
      new String'("HT")
   );

   ---------------------------------------------------------------------------
   -- Check_Abbreviation_And_Prettify
   ---------------------------------------------------------------------------
   -- Purpose:
   --  Scans through the abbreviations and sees if this subtoken is on the
   --  list.  If so, overwrites with the approved capitalization.  Otherwise,
   --  just capitalizes the first letter.
   -- Implementation Notes:
   --  Since by definition if Token = Abbreviation, they're the same length
   --  and we can use a simple assignment.
   ---------------------------------------------------------------------------
   procedure Check_Abbreviation_And_Prettify (Token : in out String) is
      Abbreviation : String_Access;
      use Ada.Strings.Maps.Constants; -- Lower_Case_Map
   begin
      for Each_Abbreviation in Program_Abbreviations'RANGE loop
         Abbreviation :=
            new String'(Program_Abbreviations(Each_Abbreviation).all);
         Ada.Strings.Fixed.Translate (Abbreviation.all, Lower_Case_Map);
         if Token = Abbreviation.all then
            Token := Program_Abbreviations(Each_Abbreviation).all;
            return;
         end if;
      end loop;
      -- Not an abbreviation, just capitalize the first letter
      Token(Token'FIRST) :=
         Ada.Characters.Handling.To_Upper (Token(Token'FIRST));
   end Check_Abbreviation_And_Prettify;


   ---------------------------------------------------------------------------
   -- Convert_To_Mixed_Case
   ---------------------------------------------------------------------------
   -- Implementation Notes:
   --  The token supplied is assumed to be in lower case
   ---------------------------------------------------------------------------
   procedure Convert_To_Mixed_Case (Token : in out String) is
      Subtoken       : String_Access;
      Token_Ptr      : Positive := Token'FIRST;
      Token_Last     : Natural := Token'LAST;
      Subtoken_Start : Positive;
      Subtoken_End   : Natural;
   begin
      while Token_Ptr <= Token_Last loop
         Ada.Strings.Fixed.Find_Token (
            Source => Token(Token_Ptr..Token_Last),
            Set    => Ada.Strings.Maps.Constants.Alphanumeric_Set,
            Test   => Ada.Strings.Inside,
            First  => Subtoken_Start,
            Last   => Subtoken_End);
         Subtoken := new String'(Token(Subtoken_Start..Subtoken_End));
         Check_Abbreviation_And_Prettify (Subtoken.all);
         Ada.Strings.Fixed.Overwrite (Token, Subtoken_Start, Subtoken.all);
         Token_Ptr := Subtoken_End + 1;
      end loop;
   end Convert_To_Mixed_Case;

end Abbreviations;
