------------------------------------------------------------------------------
-- Unit Name:   Prettify
-- Unit Type:   main procedure
-- Source File: prettify.adb
-- Author:      Bob Crispen <crispen@hiwaay.net>
-- Date:        19 April 1996
-- Purpose:
--  Convert one or more lines of Ada 95 source code in stdin into "prettified"
--  Ada 95 source code in stdout.  Prettification involves
--    (a) Surrounding arithmetic and logical operators and colons with
--        blanks
--    (b) Removing trailing blanks in the line
--    (c) Removing unnecessary blanks before ')' and after '('
--    (d) Putting Ada keywords in lower case
--    (e) Putting Ada attribute names in upper case
--    (f) Putting everything else in mixed case
--    (g) Leaving string literals and comments alone
-- Implementation Notes:
--  Gnat 3.01a's Trim() doesn't work, so we include a substitute here
--
-- External Unit Declarations:
--
with Keywords;
with Abbreviations;
with Ada.Characters.Latin_1;
with Ada.Characters.Handling;
with Ada.Text_IO;
with IO_Exceptions;
with Ada.Strings.Fixed;
with Ada.Strings.Maps.Constants;
--
-- Visibility Declarations:
--
use Ada.Characters.Latin_1;      -- cr, ht, lf
use Ada.Strings.Maps.Constants;  -- Lower_Case_Map, Upper_Case_Map
--
-- Unit Declaration:
--
procedure Prettify is
--
-- Exceptions Raised:
--  End_Error (handled)
--
-- This software is provided under the terms of the Gnu General Public
-- License
------------------------------------------------------------------------------

   Max_String_Length : constant := 1000;
   A_Little_Bit      : constant := 100;

   subtype Bufs is String(1..Max_String_Length+A_Little_Bit);

   Whitespace_Set : constant Ada.Strings.Maps.Character_Set :=
      Ada.Strings.Maps.To_Set (' ' & cr & ht & lf);

   Name_Character_Set : constant Ada.Strings.Maps.Character_Set :=
      Ada.Strings.Maps.To_Set (
      "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz_");

   type String_Access is access String;

   Input_Buf : String(1..Max_String_Length);
   Last      : Natural;

   ---------------------------------------------------------------------------
   -- Is_Whitespace
   ---------------------------------------------------------------------------
   -- Purpose:
   --  Notational abbreviation
   --  Blank, Tab, CR and LF are our whitespace characters
   -- Implementation Notes:
   ---------------------------------------------------------------------------
   function Is_Whitespace (The_Character : in Character) return Boolean is
   begin
      return Ada.Strings.Maps.Is_In (The_Character, Whitespace_Set);
   end Is_Whitespace;
   pragma Inline (Is_Whitespace);


   ---------------------------------------------------------------------------
   -- Last_Non_Whitespace_Character
   ---------------------------------------------------------------------------
   -- Purpose:
   --  Find the last non-blank character in a string
   --  Blank, Tab, CR and LF are our whitespace characters
   -- Implementation Notes:
   --  This is here because Gnat 3.01a's Trim() is broken.
   ---------------------------------------------------------------------------
   function Last_Non_Whitespace_Character_In (The_Line : String)
    return Natural is
   begin
      for Each_Character in reverse The_Line'RANGE loop
         if not Is_Whitespace (The_Line(Each_Character)) then
            return Each_Character;
         end if;
      end loop;
      return 0;
   end Last_Non_Whitespace_Character_In;


   ---------------------------------------------------------------------------
   -- Prettify_One
   ---------------------------------------------------------------------------
   -- Purpose:
   --  Prettify a single line of text
   -- Implementation Notes:
   --   On entry, the line has had all trailing whitespace characters
   --   removed
   ---------------------------------------------------------------------------
   procedure Prettify_One (Line : in String) is
      Hold                : Bufs;
      Output_Line         : Bufs;
      Token               : String_Access;
      Token_Start         : Positive := 1;
      Token_End           : Natural  := 0;
      May_Be_An_Attribute : Boolean  := False;
      Hold_Ptr            : Positive := 1;
      Last_In_Hold        : Natural  := Line'LAST;
      This_Char           : Character;
      Prev_Char           : Character := ' ';
      In_Comment          : Boolean  := False;
      In_String           : Boolean  := False;
      Dont_Copy_Current   : Boolean := False;

      ------------------------------------------------------------------------
      -- Add_Space
      ------------------------------------------------------------------------
      -- Purpose:
      --  Add a space to the hold buffer
      -- Implementation Notes:
      --  Modifies variables Hold_Ptr and Hold
      ------------------------------------------------------------------------
      procedure Add_Space is
      begin
         Hold(Hold_Ptr) := ' ';
         Hold_Ptr := Hold_Ptr + 1;
      end Add_Space;
      pragma Inline (Add_Space);

      ------------------------------------------------------------------------
      -- Check_Comment_And_String
      ------------------------------------------------------------------------
      -- Purpose:
      --  See if we are entering a comment or string, or exiting a string
      -- Implementation Notes:
      --  Reads variables This_Char and Prev_Char
      --  Modifies variables In_String and In_Comment
      ------------------------------------------------------------------------
      procedure Check_Comment_And_String is
      begin
         if This_Char = '-' and Prev_Char = '-' then
            In_Comment := True;
         end if;

         if In_String then
            In_String := This_Char /= '"';
         else
            In_String := This_Char = '"';
         end if;
      end Check_Comment_And_String;
      pragma Inline (Check_Comment_And_String);

   begin

      -- Ada.Text_IO.Put_Line ("Line = <" & Line & ">");

      ------------------------------------------------------------------------
      -- Copy the input string to a hold buffer, using rules that appear
      -- beside the code that implements them
      ------------------------------------------------------------------------
      for Each_Character in Line'RANGE loop
         This_Char := Line (Each_Character);

         -- If we're inside a string or inside a comment, just copy the
         -- rest of the line verbatim
         if In_Comment or In_String then
            Hold(Hold_Ptr) := This_Char;
            Hold_Ptr := Hold_Ptr + 1;
         else

            ------------------------------------------------------------------
            -- Surround arithmetic operators, comparison operators, and
            -- colons with blanks
	    --
	    -- Ada has some two-character operators:
	    --  =>, >=, <=, :=, --, /=
	    -- that make this logic look a little ugly.  On the other hand,
	    -- it's a whole lot faster than comparing and inserting strings.
	    --
            -- XXX Some folks have a rule that requires
            --  Y := (X + 1) / 2;
            --  Y := Sin(X + 1) / 2;
            --  My_Array(X+1) := 0;
            -- Without parsing for semantics, I don't have a clue how to
            -- do that.
            ------------------------------------------------------------------
            -- Leading blank
            if not Is_Whitespace (Prev_Char) and
               Prev_Char /= Apostrophe and
               ((This_Char = '=' and
                  Prev_Char /= ':' and
                  Prev_Char /= '/' and
                  Prev_Char /= '<' and
                  Prev_Char /= '>') or
                (This_Char = '-' and Prev_Char /= '-') or
                (This_Char = '>' and Prev_Char /= '=') or
                This_Char = '/' or
                This_Char = '*' or
                This_Char = '+' or
                This_Char = '<' or
                This_Char = ':')
             then
               Add_Space;
            end if;
            -- Trailing blank
            if not Is_Whitespace (This_Char) and
               This_Char /= Apostrophe and
               ((Prev_Char = '-' and This_Char /= '-') or
                 Prev_Char = '*' or
                 Prev_Char = '+' or
                 (Prev_Char = '=' and This_Char /= '>') or
                 ((Prev_Char = '/' or
                   Prev_Char = ':' or
                   Prev_Char = '>' or
                   Prev_Char = '<') and
                 This_Char /= '='))
             then
               Add_Space;
            end if;

            -- Left parens and periods should not be followed by a space
            if (Prev_Char = '(' or Prev_Char = '.') and
               Is_Whitespace (This_Char) and
               Each_Character /= Line'LAST
             then
               This_Char := Prev_Char;
               Dont_Copy_Current := True;
            end if;

            -- Right parens, commas, periods, and semicolons should not
            -- be preceded by a space
            if This_Char = ';' or This_Char = ')' or This_Char = ',' or
               This_Char = '.'
             then
               while (Is_Whitespace (Prev_Char) and Hold_Ptr > 1) loop
                  Hold_Ptr := Hold_Ptr - 1;
                  if Hold_Ptr > 1 then
                     Prev_Char := Hold(Hold_Ptr-1);
                  else
                     Prev_Char := ' ';
                  end if;
               end loop;
            end if;

            -- Unless the current character is a space following a left
            -- paren, copy the current character
            if Dont_Copy_Current then
               Dont_Copy_Current := False;
            else
               Hold(Hold_Ptr) := This_Char;
               Hold_Ptr := Hold_Ptr + 1;
            end if;

         end if;

         Check_Comment_And_String;
         Prev_Char := This_Char;

      end loop;

      -- Now the string is copied.  Set the limits for the buffer we just
      -- copied it to, and get ready to convert the case of the characters
      -- in the line.
      Last_In_Hold := Hold_Ptr - 1;
      Hold_Ptr     := 1;
      -- Ada.Text_IO.Put_Line ("Hold = <" & Hold(1..Last_In_Hold) & ">");
      In_Comment := False;
      In_String  := False;
      Prev_Char := ' ';
      This_Char := ' ';

      ------------------------------------------------------------------------
      -- Process characters out of the hold buffer into the output buffer,
      -- converting case as necessary
      -- XXX Some folks who have coded *entirely* too much C require
      -- constants and enumeration values to be in upper-case.  I consider
      -- this (a) bad practice, and (b) too hard for this program to
      -- implement.
      ------------------------------------------------------------------------
      loop
         This_Char := Hold(Hold_Ptr);
         if In_Comment or In_String then
            Output_Line(Hold_Ptr) := This_Char;
            Hold_Ptr := Hold_Ptr + 1;
         else
            if Ada.Characters.Handling.Is_Letter (This_Char) then
               Ada.Strings.Fixed.Find_Token (
                  Source => Hold(Hold_Ptr..Last_In_Hold),
                  Set    => Name_Character_Set,
                  Test   => Ada.Strings.Inside,
                  First  => Token_Start,
                  Last   => Token_End);
               Token := new String'(Hold(Token_Start..Token_End));
               -- Ada.Text_IO.Put_Line ("Token = <" & Token.all & ">");
               Ada.Strings.Fixed.Translate (Token.all, Lower_Case_Map);
               if May_Be_An_Attribute then
                  if Keywords.Is_Attribute (Token.all) then
                     Ada.Strings.Fixed.Translate (Token.all, Upper_Case_Map);
                     -- Ada.Text_IO.Put_Line ("Found " & Token.all);
                  elsif not Keywords.Is_Reserved_Word (Token.all) then
                     Abbreviations.Convert_To_Mixed_Case (Token.all);
                  end if;
               else
                  if not Keywords.Is_Reserved_Word (Token.all) then
                     Abbreviations.Convert_To_Mixed_Case (Token.all);
                  end if;
               end if;
               Ada.Strings.Fixed.Overwrite (Output_Line, Hold_Ptr, Token.all);
               May_Be_An_Attribute  := False;
               Hold_Ptr := Hold_Ptr + Token'LENGTH;
            else -- not an alpha character, copy one at a time
               Output_Line (Hold_Ptr) := This_Char;
               if This_Char = Apostrophe then
                  May_Be_An_Attribute := True;
               elsif not Is_Whitespace (This_Char) then
                  May_Be_An_Attribute := False;
                  -- but if it's a whitespace, postpone judgement on
                  -- whether an attribute is next
               end if;
               Hold_Ptr := Hold_Ptr + 1;
            end if;
         end if;
         Check_Comment_And_String;
         Prev_Char := This_Char;
         exit when Hold_Ptr > Last_In_Hold;
      end loop;
      Ada.Text_IO.Put_Line (Output_Line(1..Last_In_Hold));
   end Prettify_One;

------------------------------------------------------------------------------
-- Main procedure
------------------------------------------------------------------------------
begin
   loop
      Ada.Text_IO.Get_Line (Input_Buf, Last);
      -- Use Ada.Characters.Fixed.Trim when it's fixed
      Last := Last_Non_Whitespace_Character_In (Input_Buf (1..Last));
      Prettify_One (Line => Input_Buf(1..Last));
   end loop;
exception
   when Ada.IO_Exceptions.End_Error =>
      null;
end Prettify;
