/*
 * A little program to figure out the single axis+angle that
 * corresponds to a series of rotations.
 *
 * Written by Gavin Bell.
 */

#include <stdio.h>
#include <math.h>

/*
 * Convert axis/angle to quaternion form:
 * Assumes axis is unit-length, angle is in radians.
 */
void
to_quaternion(float axis[3], float angle, float result[4])
{
    int i;
    for (i = 0; i < 3; i++) {
	result[i] = axis[i]*sin(angle / 2.0);
    }
    result[3] = cos(angle / 2.0);
}
/*
 * Convert unit-length quaternion to axis/angle.
 */
void
from_quaternion(float q[4], float axis[3], float *angle)
{
    int i;
    *angle = acos(q[3]) * 2.0;
    for (i = 0; i < 3; i++) {
	axis[i] = q[i] / sin(*angle / 2.0);
    }
}

/*
 * Multiply two quaternions together.  Result may be the same as
 * either input:
 */
void
multiply_quaternion(float q1[4], float q2[4], float result[4])
{
    float tmp[4];
    tmp[0] = q2[3] * q1[0] + q2[0] * q1[3] +
	q2[1] * q1[2] - q2[2] * q1[1];
    tmp[1] = q2[3] * q1[1] + q2[1] * q1[3] +
	q2[2] * q1[0] - q2[0] * q1[2];
    tmp[2] = q2[3] * q1[2] + q2[2] * q1[3] +
	q2[0] * q1[1] - q2[1] * q1[0];
    tmp[3] = q2[3] * q1[3] - q2[0] * q1[0] -
	q2[1] * q1[1] - q2[2] * q1[2];
    result[0] = tmp[0]; result[1] = tmp[1];
    result[2] = tmp[2]; result[3] = tmp[3];
}

#define MAX_ROTATIONS 10

main(int argc, char **argv)
{
    float axis[MAX_ROTATIONS][3], angle[MAX_ROTATIONS];
    float q[4], result_axis[3], result_angle;
    int num_entered = 0;
    int i;

    printf("\nThis program allows you to enter a series of rotations\n"
	   "about the X, Y, or Z axes.  It will combine them into\n"
	   "one rotation about an arbitrary axis, suitable for\n"
	   "use in a VRML camera's orientation field or any other\n"
	   "VRML SFRotation field.\n"
	   "Enter rotations as if you are rotating the object or\n"
	   "camera (the opposite order from nested transformations in\n"
	   "a VRML file):\n\n");

    for (i = 0; i < MAX_ROTATIONS; i++) {
	char axisString[10];
	int done = 0;
	float angle_degrees;

	axis[i][0] = 0.0;
	axis[i][1] = 0.0;
	axis[i][2] = 0.0;
	angle[i] = 0.0;

	if (i == 0) {
	    printf("Axis (X, Y or Z): ");
	} else {
	    printf("Axis [XYZ or Q]:  ");
	}
	fflush(stdout);
	scanf("%s", axisString);
	switch(axisString[0]) {
	  case 'X':
	  case 'x':
	    axis[i][0] = 1.0;
	    break;
	  case 'Y':
	  case 'y':
	    axis[i][1] = 1.0;
	    break;
	  case 'Z':
	  case 'z':
	    axis[i][2] = 1.0;
	    break;
	  default:
	    done = 1;
	    break;
	}
	if (done) {
	    num_entered = i;
	    break;
	}

	printf("Angle in degrees: ");
	fflush(stdout);
	scanf("%f", &angle_degrees);
	angle[i] = angle_degrees * 3.141592653 / 180.0;
	printf("      You entered: (%g %g %g), %g\n", axis[i][0],
               axis[i][1], axis[i][2], angle[i]);
    }

    to_quaternion(axis[0], angle[0], q);
    for (i = 1; i < num_entered; i++) {
	float q1[4];

	to_quaternion(axis[i], angle[i], q1);
	multiply_quaternion(q, q1, q);
    }
    from_quaternion(q, result_axis, &result_angle);

    printf("\nCombined rotation is:\n");
    printf("Transform { rotation %g %g %g  %g }\n\n",
	   result_axis[0], result_axis[1], result_axis[2],
	   result_angle);

    return 0;
}
