 /*
 #--------------------------------------------------------------------
 #
 #  Include File: fvtmtabl.h
 #
 #  This include declares the tables used by the floating point
 #  DECtalk vocal tract model fvtm.s
 #
 #  Data tables used to speed up synthesizer calculations
 #
 #--------------------------------------------------------------------
 #--------------------------------------------------------------------
 #
 #  Constant B0 controls shape of glottal pulse as a function
 #  of desired duration of open phase N0.
 #
 #  (Note that N0 is specified in terms of 4 times the output sample
 #  rate )
 #
 #  Assume voicing waveform V(t) has form: k1 t--2 - k2 t--3
 #
 #  If the radiation characteristic, a temporal derivative is folded
 #  in, and we go from continuous time to discrete integers n:
 #
 #  dV/dt = voice0[n] = sum over i=1,2,...,n of { a - (i * b) }
 #
 #                    = a n  -  b/2 n--2
 #
 #  WHERE the constants a and b control the detailed shape and
 #  amplitude of the voicing waveform over the open portion of the
 #  voicing cycle "nopen".
 #
 #  Let integral of dV/dt have no net dc flow --> a = (b * nopen) / 3
 #
 #  Let maximum of dUg(n)/dn be constant i.e. :
 #
 #      b = gain / (nopen * nopen)
 #
 #  meaning as nopen gets bigger, V has bigger peak proportional to n
 #
 #  Thus, to generate the table below for 40 <= nopen <= 263:
 #
 #      B0[nopen - 40] = 1920000 / (nopen * nopen)
 #
 #--------------------------------------------------------------------
 */
        .data
        .align 5

B0Table:
        .s_floating  75.000000,  71.386080,  68.027211,  64.899946,  61.983471
        .s_floating  59.259259,  56.710775,  54.323223,  52.083333,  49.979175
        .s_floating  48.000000,  46.136101,  44.378698,  42.719829,  41.152263
        .s_floating  39.669421,  38.265306,  36.934441,  35.671819,  34.472853
        .s_floating  33.333333,  32.249395,  31.217482,  30.234316,  29.296875
        .s_floating  28.402367,  27.548209,  26.732012,  25.951557,  25.204789
        .s_floating  24.489796,  23.804801,  23.148148,  22.518296,  21.913806
        .s_floating  21.333333,  20.775623,  20.239501,  19.723866,  19.227688
        .s_floating  18.750000,  18.289895,  17.846520,  17.419074,  17.006803
        .s_floating  16.608997,  16.224986,  15.854142,  15.495868,  15.149602
        .s_floating  14.814815,  14.491004,  14.177694,  13.874436,  13.580806
        .s_floating  13.296399,  13.020833,  12.753746,  12.494794,  12.243649
        .s_floating  12.000000,  11.763553,  11.534025,  11.311151,  11.094675
        .s_floating  10.884354,  10.679957,  10.481265,  10.288066,  10.100160
        .s_floating  9.917355,  9.739469,  9.566327,  9.397760,  9.233610
        .s_floating  9.073724,  8.917955,  8.766163,  8.618213,  8.473978
        .s_floating  8.333333,  8.196161,  8.062349,  7.931787,  7.804370
        .s_floating  7.680000,  7.558579,  7.440015,  7.324219,  7.211105
        .s_floating  7.100592,  6.992599,  6.887052,  6.783877,  6.683003
        .s_floating  6.584362,  6.487889,  6.393521,  6.301197,  6.210859
        .s_floating  6.122449,  6.035914,  5.951200,  5.868258,  5.787037
        .s_floating  5.707491,  5.629574,  5.553242,  5.478451,  5.405162
        .s_floating  5.333333,  5.262927,  5.193906,  5.126233,  5.059875
        .s_floating  4.994797,  4.930966,  4.868352,  4.806922,  4.746648
        .s_floating  4.687500,  4.629451,  4.572474,  4.516542,  4.461630
        .s_floating  4.407713,  4.354768,  4.302772,  4.251701,  4.201534
        .s_floating  4.152249,  4.103827,  4.056247,  4.009489,  3.963535
        .s_floating  3.918367,  3.873967,  3.830317,  3.787401,  3.745201
        .s_floating  3.703704,  3.662892,  3.622751,  3.583266,  3.544423
        .s_floating  3.506209,  3.468609,  3.431611,  3.395201,  3.359368
        .s_floating  3.324100,  3.289384,  3.255208,  3.221563,  3.188437
        .s_floating  3.155819,  3.123698,  3.092066,  3.060912,  3.030227
        .s_floating  3.000000,  2.970224,  2.940888,  2.911985,  2.883506
        .s_floating  2.855443,  2.827788,  2.800532,  2.773669,  2.747190
        .s_floating  2.721088,  2.695357,  2.669989,  2.644978,  2.620316
        .s_floating  2.595998,  2.572016,  2.548366,  2.525040,  2.502033
        .s_floating  2.479339,  2.456952,  2.434867,  2.413079,  2.391582
        .s_floating  2.370370,  2.349440,  2.328786,  2.308403,  2.288286
        .s_floating  2.268431,  2.248833,  2.229489,  2.210393,  2.191541
        .s_floating  2.172929,  2.154553,  2.136410,  2.118494,  2.100804
        .s_floating  2.083333,  2.066080,  2.049040,  2.032211,  2.015587
        .s_floating  1.999167,  1.982947,  1.966923,  1.951093,  1.935453
        .s_floating  1.920000,  1.904732,  1.889645,  1.874736,  1.860004
        .s_floating  1.845444,  1.831055,  1.816833,  1.802776,  1.788882
        .s_floating  1.775148,  1.761571,  1.748150,  1.734881

 /*
 #--------------------------------------------------------------------
 #
 #  Data arrays for conversion of nasal antiresonator FZinHZ & BW to
 #  difference equation coeficients b0, b1, and b2.
 #
 #  The arrays Nasal_b0_Table[FZinHZ/8],
 #             Nasal_b1_Table[FZinHZ/8],
 #             Nasal_b2_Table[FZinHZ/8],
 #
 #  were computed from the equations:
 #
 #    a1 = - exp( - 2 pi BW T )
 #    a2 = 2 exp( - pi BW T) * cos( 2 pi FZinHZ T )
 #    b0_T = 1.0 - a1 - a2
 #
 #      b0 = Gain / b0_T
 #      b1 = -a1 * b0
 #      b2 = -a2 * b0
 #
 #  where  FZinHZ = 0, 8, 16, ..., 4992 Hz
 #  and BW was fixed at 80 Hz, T = 1/10000, and Gain = 1.0.
 #  and the gain factor Gain is 1.0
 #
 #--------------------------------------------------------------------
 */
        .data
        .align 5

Nasal_b0_Table:
        .s_floating  1623.350994,  1560.917704,  1399.452298,  1193.663055,  989.881513
        .s_floating  811.718222,  665.358214,  548.485973,  456.058920,  382.931102
        .s_floating  324.738565,  278.041887,  240.213500,  209.269167,  183.712790
        .s_floating  162.412024,  144.504115,  129.326363,  116.365247,  105.219262
        .s_floating  95.571696,  87.170554,  79.813667,  73.337557,  67.609032
        .s_floating  62.518802,  57.976585,  53.907325,  50.248247,  46.946550
        .s_floating  43.957583,  41.243397,  38.771579,  36.514322,  34.447668
        .s_floating  32.550888,  30.805977,  29.197241,  27.710950,  26.335052
        .s_floating  25.058934,  23.873222,  22.769614,  21.740734,  20.780013
        .s_floating  19.881585,  19.040199,  18.251144,  17.510184,  16.813502
        .s_floating  16.157651,  15.539511,  14.956253,  14.405310,  13.884345
        .s_floating  13.391227,  12.924012,  12.480921,  12.060326,  11.660733
        .s_floating  11.280769,  10.919173,  10.574783,  10.246525,  9.933410
        .s_floating  9.634523,  9.349016,  9.076104,  8.815057,  8.565200
        .s_floating  8.325903,  8.096579,  7.876684,  7.665709,  7.463177
        .s_floating  7.268646,  7.081700,  6.901950,  6.729032,  6.562605
        .s_floating  6.402348,  6.247961,  6.099161,  5.955681,  5.817272
        .s_floating  5.683697,  5.554735,  5.430176,  5.309821,  5.193484
        .s_floating  5.080989,  4.972168,  4.866863,  4.764924,  4.666210
        .s_floating  4.570586,  4.477925,  4.388106,  4.301014,  4.216539
        .s_floating  4.134578,  4.055033,  3.977809,  3.902818,  3.829973
        .s_floating  3.759194,  3.690404,  3.623529,  3.558498,  3.495245
        .s_floating  3.433706,  3.373818,  3.315524,  3.258768,  3.203496
        .s_floating  3.149656,  3.097201,  3.046082,  2.996255,  2.947677
        .s_floating  2.900307,  2.854104,  2.809031,  2.765051,  2.722130
        .s_floating  2.680234,  2.639329,  2.599387,  2.560375,  2.522267
        .s_floating  2.485033,  2.448649,  2.413087,  2.378324,  2.344336
        .s_floating  2.311100,  2.278593,  2.246796,  2.215687,  2.185247
        .s_floating  2.155456,  2.126297,  2.097752,  2.069804,  2.042435
        .s_floating  2.015631,  1.989376,  1.963655,  1.938454,  1.913758
        .s_floating  1.889555,  1.865832,  1.842575,  1.819774,  1.797415
        .s_floating  1.775488,  1.753981,  1.732885,  1.712188,  1.691881
        .s_floating  1.671954,  1.652397,  1.633202,  1.614360,  1.595862
        .s_floating  1.577700,  1.559865,  1.542350,  1.525148,  1.508251
        .s_floating  1.491652,  1.475344,  1.459319,  1.443573,  1.428098
        .s_floating  1.412888,  1.397938,  1.383240,  1.368791,  1.354583
        .s_floating  1.340613,  1.326874,  1.313361,  1.300070,  1.286996
        .s_floating  1.274135,  1.261481,  1.249030,  1.236778,  1.224720
        .s_floating  1.212854,  1.201174,  1.189677,  1.178359,  1.167216
        .s_floating  1.156245,  1.145443,  1.134805,  1.124329,  1.114011
        .s_floating  1.103849,  1.093838,  1.083977,  1.074262,  1.064691
        .s_floating  1.055260,  1.045966,  1.036809,  1.027783,  1.018888
        .s_floating  1.010121,  1.001478,  0.992959,  0.984560,  0.976280
        .s_floating  0.968116,  0.960066,  0.952127,  0.944299,  0.936579
        .s_floating  0.928964,  0.921454,  0.914046,  0.906738,  0.899528
        .s_floating  0.892416,  0.885399,  0.878475,  0.871643,  0.864901
        .s_floating  0.858248,  0.851682,  0.845202,  0.838806,  0.832493
        .s_floating  0.826261,  0.820110,  0.814037,  0.808041,  0.802121
        .s_floating  0.796277,  0.790505,  0.784806,  0.779179,  0.773621
        .s_floating  0.768133,  0.762712,  0.757357,  0.752069,  0.746845
        .s_floating  0.741684,  0.736587,  0.731550,  0.726575,  0.721659
        .s_floating  0.716801,  0.712002,  0.707260,  0.702573,  0.697942
        .s_floating  0.693365,  0.688842,  0.684371,  0.679952,  0.675585
        .s_floating  0.671268,  0.667000,  0.662782,  0.658611,  0.654488
        .s_floating  0.650412,  0.646382,  0.642398,  0.638458,  0.634562
        .s_floating  0.630710,  0.626900,  0.623133,  0.619408,  0.615723
        .s_floating  0.612079,  0.608475,  0.604910,  0.601384,  0.597897
        .s_floating  0.594447,  0.591034,  0.587658,  0.584318,  0.581014
        .s_floating  0.577745,  0.574511,  0.571311,  0.568145,  0.565013
        .s_floating  0.561913,  0.558846,  0.555810,  0.552807,  0.549834
        .s_floating  0.546893,  0.543981,  0.541100,  0.538248,  0.535426
        .s_floating  0.532632,  0.529867,  0.527130,  0.524421,  0.521739
        .s_floating  0.519085,  0.516456,  0.513855,  0.511279,  0.508729
        .s_floating  0.506205,  0.503705,  0.501231,  0.498781,  0.496355
        .s_floating  0.493952,  0.491574,  0.489219,  0.486887,  0.484577
        .s_floating  0.482290,  0.480025,  0.477782,  0.475561,  0.473361
        .s_floating  0.471183,  0.469025,  0.466888,  0.464772,  0.462675
        .s_floating  0.460599,  0.458542,  0.456505,  0.454487,  0.452488
        .s_floating  0.450508,  0.448546,  0.446603,  0.444678,  0.442772
        .s_floating  0.440882,  0.439011,  0.437157,  0.435320,  0.433500
        .s_floating  0.431697,  0.429911,  0.428141,  0.426387,  0.424650
        .s_floating  0.422928,  0.421222,  0.419532,  0.417857,  0.416198
        .s_floating  0.414553,  0.412924,  0.411309,  0.409709,  0.408123
        .s_floating  0.406552,  0.404994,  0.403451,  0.401922,  0.400406
        .s_floating  0.398904,  0.397416,  0.395940,  0.394478,  0.393029
        .s_floating  0.391593,  0.390170,  0.388759,  0.387361,  0.385975
        .s_floating  0.384601,  0.383240,  0.381891,  0.380553,  0.379227
        .s_floating  0.377913,  0.376611,  0.375320,  0.374040,  0.372772
        .s_floating  0.371515,  0.370268,  0.369033,  0.367809,  0.366595
        .s_floating  0.365392,  0.364199,  0.363017,  0.361845,  0.360683
        .s_floating  0.359531,  0.358390,  0.357258,  0.356136,  0.355024
        .s_floating  0.353921,  0.352829,  0.351745,  0.350671,  0.349607
        .s_floating  0.348551,  0.347505,  0.346468,  0.345439,  0.344420
        .s_floating  0.343410,  0.342408,  0.341415,  0.340431,  0.339455
        .s_floating  0.338487,  0.337528,  0.336578,  0.335635,  0.334701
        .s_floating  0.333775,  0.332857,  0.331946,  0.331044,  0.330150
        .s_floating  0.329263,  0.328384,  0.327513,  0.326649,  0.325793
        .s_floating  0.324944,  0.324103,  0.323269,  0.322442,  0.321623
        .s_floating  0.320810,  0.320005,  0.319207,  0.318416,  0.317631
        .s_floating  0.316854,  0.316084,  0.315320,  0.314563,  0.313813
        .s_floating  0.313069,  0.312332,  0.311601,  0.310877,  0.310160
        .s_floating  0.309448,  0.308744,  0.308045,  0.307353,  0.306666
        .s_floating  0.305986,  0.305312,  0.304645,  0.303983,  0.303327
        .s_floating  0.302677,  0.302033,  0.301395,  0.300763,  0.300136
        .s_floating  0.299515,  0.298900,  0.298291,  0.297687,  0.297089
        .s_floating  0.296496,  0.295909,  0.295327,  0.294751,  0.294180
        .s_floating  0.293614,  0.293054,  0.292499,  0.291949,  0.291405
        .s_floating  0.290866,  0.290331,  0.289802,  0.289279,  0.288760
        .s_floating  0.288246,  0.287737,  0.287233,  0.286734,  0.286241
        .s_floating  0.285751,  0.285267,  0.284788,  0.284313,  0.283843
        .s_floating  0.283378,  0.282918,  0.282462,  0.282011,  0.281565
        .s_floating  0.281123,  0.280686,  0.280253,  0.279825,  0.279401
        .s_floating  0.278982,  0.278567,  0.278157,  0.277751,  0.277350
        .s_floating  0.276953,  0.276560,  0.276172,  0.275788,  0.275408
        .s_floating  0.275032,  0.274661,  0.274294,  0.273931,  0.273572
        .s_floating  0.273217,  0.272867,  0.272520,  0.272178,  0.271840
        .s_floating  0.271506,  0.271176,  0.270850,  0.270527,  0.270209
        .s_floating  0.269895,  0.269585,  0.269279,  0.268976,  0.268678
        .s_floating  0.268383,  0.268093,  0.267806,  0.267523,  0.267244
        .s_floating  0.266968,  0.266697,  0.266429,  0.266165,  0.265905
        .s_floating  0.265648,  0.265395,  0.265146,  0.264901,  0.264659
        .s_floating  0.264421,  0.264187,  0.263956,  0.263729,  0.263505
        .s_floating  0.263285,  0.263069,  0.262856,  0.262647,  0.262442
        .s_floating  0.262240,  0.262041,  0.261846,  0.261655,  0.261467
        .s_floating  0.261282,  0.261101,  0.260924,  0.260750,  0.260579
        .s_floating  0.260412,  0.260249,  0.260089,  0.259932,  0.259778
        .s_floating  0.259629,  0.259482,  0.259339,  0.259199,  0.259063
        .s_floating  0.258930,  0.258800,  0.258674,  0.258551,  0.258432
        .s_floating  0.258316,  0.258203,  0.258093,  0.257987,  0.257884
        .s_floating  0.257785,  0.257689,  0.257596,  0.257506,  0.257420
        .s_floating  0.257337,  0.257257,  0.257181,  0.257107,  0.257038
        .s_floating  0.256971,  0.256908,  0.256848,  0.256791,  0.256737
        .s_floating  0.256687,  0.256640,  0.256596,  0.256556,  0.256518
        .s_floating  0.256484,  0.256453,  0.256426,  0.256402,  0.256381
        .s_floating  0.256363,  0.256348,  0.256337,  0.256329,  0.256324

Nasal_b1_Table:
        .s_floating  -3166.120328,  -3044.314419,  -2729.299138,  -2327.809075,  -1930.235988
        .s_floating  -1582.643519,  -1297.098522,  -1069.083476,  -888.760429,  -746.089742
        .s_floating  -632.557447,  -541.453306,  -467.650995,  -407.279316,  -357.419414
        .s_floating  -315.862110,  -280.924194,  -251.312750,  -226.025911,  -204.280353
        .s_floating  -185.458174,  -169.067739,  -154.714623,  -142.079881,  -130.903661
        .s_floating  -120.972740,  -112.110980,  -104.171947,  -97.033170,  -90.591636
        .s_floating  -84.760231,  -79.464915,  -74.642455,  -70.238600,  -66.206606
        .s_floating  -62.506031,  -59.101750,  -55.963144,  -53.063424,  -50.379078
        .s_floating  -47.889401,  -45.576105,  -43.422992,  -41.415671,  -39.541326
        .s_floating  -37.788513,  -36.146988,  -34.607560,  -33.161965,  -31.802755
        .s_floating  -30.523204,  -29.317227,  -28.179305,  -27.104428,  -26.088037
        .s_floating  -25.125975,  -24.214449,  -23.349989,  -22.529418,  -21.749820
        .s_floating  -21.008520,  -20.303055,  -19.631157,  -18.990734,  -18.379855
        .s_floating  -17.796732,  -17.239715,  -16.707269,  -16.197973,  -15.710507
        .s_floating  -15.243644,  -14.796239,  -14.367229,  -13.955621,  -13.560486
        .s_floating  -13.180960,  -12.816232,  -12.465544,  -12.128186,  -11.803490
        .s_floating  -11.490833,  -11.189627,  -10.899321,  -10.619396,  -10.349363
        .s_floating  -10.088762,  -9.837160,  -9.594148,  -9.359339,  -9.132368
        .s_floating  -8.912892,  -8.700585,  -8.495137,  -8.296257,  -8.103668
        .s_floating  -7.917109,  -7.736329,  -7.561094,  -7.391179,  -7.226370
        .s_floating  -7.066467,  -6.911276,  -6.760614,  -6.614307,  -6.472189
        .s_floating  -6.334101,  -6.199893,  -6.069421,  -5.942548,  -5.819143
        .s_floating  -5.699081,  -5.582241,  -5.468511,  -5.357781,  -5.249946
        .s_floating  -5.144907,  -5.042567,  -4.942836,  -4.845625,  -4.750850
        .s_floating  -4.658431,  -4.568291,  -4.480355,  -4.394552,  -4.310813
        .s_floating  -4.229074,  -4.149271,  -4.071343,  -3.995233,  -3.920884
        .s_floating  -3.848243,  -3.777257,  -3.707877,  -3.640055,  -3.573745
        .s_floating  -3.508902,  -3.445483,  -3.383447,  -3.322754,  -3.263366
        .s_floating  -3.205246,  -3.148357,  -3.092666,  -3.038139,  -2.984744
        .s_floating  -2.932450,  -2.881227,  -2.831046,  -2.781879,  -2.733699
        .s_floating  -2.686479,  -2.640195,  -2.594822,  -2.550336,  -2.506715
        .s_floating  -2.463935,  -2.421977,  -2.380818,  -2.340439,  -2.300821
        .s_floating  -2.261943,  -2.223789,  -2.186340,  -2.149579,  -2.113490
        .s_floating  -2.078056,  -2.043261,  -2.009090,  -1.975529,  -1.942563
        .s_floating  -1.910178,  -1.878361,  -1.847099,  -1.816378,  -1.786186
        .s_floating  -1.756512,  -1.727344,  -1.698670,  -1.670479,  -1.642760
        .s_floating  -1.615504,  -1.588700,  -1.562337,  -1.536407,  -1.510900
        .s_floating  -1.485807,  -1.461119,  -1.436828,  -1.412925,  -1.389401
        .s_floating  -1.366250,  -1.343462,  -1.321032,  -1.298951,  -1.277212
        .s_floating  -1.255808,  -1.234732,  -1.213978,  -1.193540,  -1.173410
        .s_floating  -1.153583,  -1.134053,  -1.114814,  -1.095861,  -1.077187
        .s_floating  -1.058787,  -1.040656,  -1.022789,  -1.005181,  -0.987827
        .s_floating  -0.970722,  -0.953861,  -0.937240,  -0.920854,  -0.904700
        .s_floating  -0.888771,  -0.873066,  -0.857578,  -0.842306,  -0.827243
        .s_floating  -0.812388,  -0.797735,  -0.783282,  -0.769024,  -0.754959
        .s_floating  -0.741083,  -0.727393,  -0.713884,  -0.700555,  -0.687402
        .s_floating  -0.674423,  -0.661613,  -0.648970,  -0.636492,  -0.624175
        .s_floating  -0.612017,  -0.600015,  -0.588167,  -0.576469,  -0.564920
        .s_floating  -0.553517,  -0.542258,  -0.531139,  -0.520160,  -0.509317
        .s_floating  -0.498609,  -0.488033,  -0.477587,  -0.467269,  -0.457077
        .s_floating  -0.447009,  -0.437063,  -0.427238,  -0.417530,  -0.407939
        .s_floating  -0.398463,  -0.389099,  -0.379847,  -0.370704,  -0.361668
        .s_floating  -0.352739,  -0.343914,  -0.335192,  -0.326571,  -0.318050
        .s_floating  -0.309628,  -0.301302,  -0.293072,  -0.284935,  -0.276892
        .s_floating  -0.268939,  -0.261077,  -0.253303,  -0.245616,  -0.238016
        .s_floating  -0.230500,  -0.223068,  -0.215718,  -0.208450,  -0.201262
        .s_floating  -0.194152,  -0.187121,  -0.180166,  -0.173287,  -0.166483
        .s_floating  -0.159752,  -0.153094,  -0.146508,  -0.139992,  -0.133545
        .s_floating  -0.127168,  -0.120858,  -0.114615,  -0.108438,  -0.102327
        .s_floating  -0.096279,  -0.090295,  -0.084373,  -0.078513,  -0.072714
        .s_floating  -0.066975,  -0.061295,  -0.055674,  -0.050110,  -0.044604
        .s_floating  -0.039154,  -0.033759,  -0.028419,  -0.023133,  -0.017901
        .s_floating  -0.012722,  -0.007595,  -0.002519,  0.002506,  0.007481
        .s_floating  0.012406,  0.017283,  0.022110,  0.026891,  0.031624
        .s_floating  0.036310,  0.040951,  0.045546,  0.050096,  0.054601
        .s_floating  0.059063,  0.063482,  0.067858,  0.072191,  0.076483
        .s_floating  0.080733,  0.084943,  0.089112,  0.093242,  0.097332
        .s_floating  0.101383,  0.105395,  0.109370,  0.113307,  0.117207
        .s_floating  0.121070,  0.124897,  0.128687,  0.132443,  0.136163
        .s_floating  0.139848,  0.143500,  0.147117,  0.150701,  0.154251
        .s_floating  0.157769,  0.161254,  0.164707,  0.168128,  0.171518
        .s_floating  0.174877,  0.178205,  0.181502,  0.184770,  0.188008
        .s_floating  0.191216,  0.194395,  0.197546,  0.200668,  0.203761
        .s_floating  0.206827,  0.209865,  0.212876,  0.215860,  0.218817
        .s_floating  0.221747,  0.224651,  0.227529,  0.230382,  0.233209
        .s_floating  0.236011,  0.238788,  0.241540,  0.244268,  0.246972
        .s_floating  0.249652,  0.252308,  0.254940,  0.257550,  0.260136
        .s_floating  0.262700,  0.265241,  0.267759,  0.270256,  0.272730
        .s_floating  0.275183,  0.277615,  0.280025,  0.282414,  0.284782
        .s_floating  0.287129,  0.289456,  0.291763,  0.294050,  0.296316
        .s_floating  0.298563,  0.300790,  0.302998,  0.305187,  0.307357
        .s_floating  0.309507,  0.311640,  0.313753,  0.315849,  0.317926
        .s_floating  0.319985,  0.322026,  0.324050,  0.326056,  0.328044
        .s_floating  0.330015,  0.331970,  0.333907,  0.335828,  0.337731
        .s_floating  0.339619,  0.341490,  0.343345,  0.345183,  0.347006
        .s_floating  0.348813,  0.350604,  0.352380,  0.354141,  0.355886
        .s_floating  0.357615,  0.359330,  0.361030,  0.362715,  0.364386
        .s_floating  0.366042,  0.367683,  0.369310,  0.370923,  0.372522
        .s_floating  0.374107,  0.375678,  0.377235,  0.378778,  0.380308
        .s_floating  0.381825,  0.383328,  0.384818,  0.386295,  0.387759
        .s_floating  0.389210,  0.390648,  0.392073,  0.393486,  0.394886
        .s_floating  0.396273,  0.397648,  0.399011,  0.400362,  0.401701
        .s_floating  0.403028,  0.404342,  0.405645,  0.406937,  0.408216
        .s_floating  0.409484,  0.410740,  0.411985,  0.413219,  0.414441
        .s_floating  0.415653,  0.416853,  0.418042,  0.419220,  0.420387
        .s_floating  0.421543,  0.422689,  0.423824,  0.424948,  0.426062
        .s_floating  0.427166,  0.428259,  0.429341,  0.430414,  0.431476
        .s_floating  0.432528,  0.433570,  0.434602,  0.435624,  0.436636
        .s_floating  0.437639,  0.438631,  0.439614,  0.440588,  0.441551
        .s_floating  0.442506,  0.443450,  0.444386,  0.445311,  0.446228
        .s_floating  0.447136,  0.448034,  0.448923,  0.449803,  0.450674
        .s_floating  0.451536,  0.452389,  0.453233,  0.454068,  0.454895
        .s_floating  0.455712,  0.456521,  0.457322,  0.458114,  0.458897
        .s_floating  0.459671,  0.460438,  0.461195,  0.461945,  0.462686
        .s_floating  0.463419,  0.464143,  0.464860,  0.465568,  0.466268
        .s_floating  0.466959,  0.467643,  0.468319,  0.468987,  0.469647
        .s_floating  0.470299,  0.470943,  0.471579,  0.472207,  0.472828
        .s_floating  0.473441,  0.474046,  0.474643,  0.475233,  0.475815
        .s_floating  0.476390,  0.476957,  0.477517,  0.478069,  0.478614
        .s_floating  0.479151,  0.479681,  0.480203,  0.480718,  0.481226
        .s_floating  0.481726,  0.482220,  0.482706,  0.483184,  0.483656
        .s_floating  0.484121,  0.484578,  0.485028,  0.485471,  0.485907
        .s_floating  0.486336,  0.486758,  0.487173,  0.487581,  0.487983
        .s_floating  0.488377,  0.488764,  0.489144,  0.489518,  0.489884
        .s_floating  0.490244,  0.490597,  0.490943,  0.491283,  0.491616
        .s_floating  0.491942,  0.492261,  0.492573,  0.492879,  0.493178
        .s_floating  0.493471,  0.493756,  0.494036,  0.494308,  0.494574
        .s_floating  0.494834,  0.495086,  0.495333,  0.495572,  0.495805
        .s_floating  0.496032,  0.496252,  0.496466,  0.496673,  0.496874
        .s_floating  0.497068,  0.497256,  0.497437,  0.497612,  0.497780
        .s_floating  0.497942,  0.498097,  0.498247,  0.498389,  0.498526
        .s_floating  0.498656,  0.498779,  0.498896,  0.499007,  0.499112
        .s_floating  0.499210,  0.499302,  0.499387,  0.499466,  0.499539
        .s_floating  0.499605,  0.499665,  0.499719,  0.499766,  0.499807
        .s_floating  0.499842,  0.499871,  0.499893,  0.499908,  0.499918

Nasal_b2_Table:
        .s_floating  1543.769334,  1484.396715,  1330.846841,  1135.146020,  941.354476
        .s_floating  771.925297,  632.740308,  521.597503,  433.701509,  364.158641
        .s_floating  308.818881,  264.411418,  228.437495,  199.010149,  174.706624
        .s_floating  154.450087,  137.420079,  122.986387,  110.660664,  100.061090
        .s_floating  90.886478,  82.897185,  75.900956,  69.742324,  64.294629
        .s_floating  59.453938,  55.134395,  51.264622,  47.784923,  44.645086
        .s_floating  41.802647,  39.221519,  36.870877,  34.724278,  32.758937
        .s_floating  30.955143,  29.295773,  27.765903,  26.352474,  25.044026
        .s_floating  23.830467,  22.702883,  21.653378,  20.674937,  19.761313
        .s_floating  18.906928,  18.106789,  17.356417,  16.651781,  15.989253
        .s_floating  15.365553,  14.777716,  14.223052,  13.699118,  13.203692
        .s_floating  12.734748,  12.290437,  11.869068,  11.469092,  11.089088
        .s_floating  10.727751,  10.383882,  10.056374,  9.744209,  9.446444
        .s_floating  9.162209,  8.890699,  8.631165,  8.382916,  8.145307
        .s_floating  7.917741,  7.699660,  7.490545,  7.289912,  7.097309
        .s_floating  6.912315,  6.734533,  6.563595,  6.399154,  6.240886
        .s_floating  6.088485,  5.941667,  5.800161,  5.663715,  5.532091
        .s_floating  5.405065,  5.282425,  5.163972,  5.049517,  4.938884
        .s_floating  4.831903,  4.728417,  4.628274,  4.531333,  4.437458
        .s_floating  4.346522,  4.258404,  4.172988,  4.090165,  4.009831
        .s_floating  3.931889,  3.856243,  3.782805,  3.711489,  3.642216
        .s_floating  3.574907,  3.509489,  3.445892,  3.384050,  3.323898
        .s_floating  3.265375,  3.208423,  3.152987,  3.099013,  3.046450
        .s_floating  2.995250,  2.945366,  2.896754,  2.849370,  2.803173
        .s_floating  2.758125,  2.714187,  2.671324,  2.629500,  2.588683
        .s_floating  2.548840,  2.509941,  2.471957,  2.434858,  2.398617
        .s_floating  2.363209,  2.328608,  2.294790,  2.261731,  2.229409
        .s_floating  2.197802,  2.166890,  2.136651,  2.107067,  2.078119
        .s_floating  2.049789,  2.022060,  1.994914,  1.968336,  1.942309
        .s_floating  1.916819,  1.891851,  1.867391,  1.843425,  1.819940
        .s_floating  1.796924,  1.774363,  1.752247,  1.730563,  1.709300
        .s_floating  1.688448,  1.667996,  1.647933,  1.628251,  1.608940
        .s_floating  1.589990,  1.571392,  1.553138,  1.535219,  1.517628
        .s_floating  1.500356,  1.483396,  1.466740,  1.450381,  1.434312
        .s_floating  1.418526,  1.403018,  1.387779,  1.372805,  1.358088
        .s_floating  1.343624,  1.329406,  1.315430,  1.301688,  1.288177
        .s_floating  1.274892,  1.261826,  1.248976,  1.236337,  1.223904
        .s_floating  1.211673,  1.199639,  1.187798,  1.176147,  1.164681
        .s_floating  1.153396,  1.142289,  1.131355,  1.120592,  1.109996
        .s_floating  1.099562,  1.089289,  1.079173,  1.069211,  1.059399
        .s_floating  1.049735,  1.040215,  1.030837,  1.021599,  1.012496
        .s_floating  1.003527,  0.994690,  0.985981,  0.977398,  0.968939
        .s_floating  0.960601,  0.952383,  0.944281,  0.936294,  0.928420
        .s_floating  0.920656,  0.913000,  0.905451,  0.898007,  0.890665
        .s_floating  0.883423,  0.876281,  0.869236,  0.862287,  0.855431
        .s_floating  0.848667,  0.841994,  0.835409,  0.828912,  0.822501
        .s_floating  0.816174,  0.809930,  0.803768,  0.797686,  0.791682
        .s_floating  0.785756,  0.779905,  0.774130,  0.768428,  0.762799
        .s_floating  0.757241,  0.751752,  0.746333,  0.740981,  0.735696
        .s_floating  0.730476,  0.725321,  0.720229,  0.715200,  0.710232
        .s_floating  0.705325,  0.700477,  0.695687,  0.690956,  0.686281
        .s_floating  0.681662,  0.677097,  0.672587,  0.668131,  0.663727
        .s_floating  0.659374,  0.655073,  0.650821,  0.646619,  0.642466
        .s_floating  0.638360,  0.634302,  0.630290,  0.626324,  0.622403
        .s_floating  0.618527,  0.614694,  0.610905,  0.607158,  0.603454
        .s_floating  0.599790,  0.596168,  0.592585,  0.589042,  0.585538
        .s_floating  0.582073,  0.578646,  0.575256,  0.571903,  0.568586
        .s_floating  0.565305,  0.562060,  0.558849,  0.555673,  0.552531
        .s_floating  0.549423,  0.546347,  0.543304,  0.540293,  0.537314
        .s_floating  0.534366,  0.531449,  0.528563,  0.525706,  0.522880
        .s_floating  0.520082,  0.517314,  0.514574,  0.511862,  0.509178
        .s_floating  0.506521,  0.503892,  0.501289,  0.498712,  0.496162
        .s_floating  0.493637,  0.491138,  0.488664,  0.486215,  0.483790
        .s_floating  0.481389,  0.479012,  0.476659,  0.474329,  0.472022
        .s_floating  0.469737,  0.467475,  0.465236,  0.463018,  0.460822
        .s_floating  0.458647,  0.456493,  0.454360,  0.452248,  0.450156
        .s_floating  0.448084,  0.446032,  0.444000,  0.441987,  0.439993
        .s_floating  0.438019,  0.436063,  0.434125,  0.432206,  0.430305
        .s_floating  0.428422,  0.426557,  0.424709,  0.422879,  0.421066
        .s_floating  0.419269,  0.417489,  0.415726,  0.413979,  0.412249
        .s_floating  0.410534,  0.408835,  0.407152,  0.405485,  0.403832
        .s_floating  0.402195,  0.400573,  0.398965,  0.397373,  0.395794
        .s_floating  0.394231,  0.392681,  0.391145,  0.389624,  0.388116
        .s_floating  0.386621,  0.385140,  0.383673,  0.382218,  0.380777
        .s_floating  0.379349,  0.377933,  0.376530,  0.375140,  0.373762
        .s_floating  0.372396,  0.371042,  0.369701,  0.368371,  0.367053
        .s_floating  0.365747,  0.364452,  0.363169,  0.361897,  0.360637
        .s_floating  0.359387,  0.358148,  0.356921,  0.355704,  0.354498
        .s_floating  0.353302,  0.352117,  0.350942,  0.349778,  0.348623
        .s_floating  0.347479,  0.346345,  0.345220,  0.344106,  0.343001
        .s_floating  0.341906,  0.340820,  0.339744,  0.338677,  0.337619
        .s_floating  0.336571,  0.335532,  0.334502,  0.333480,  0.332468
        .s_floating  0.331464,  0.330469,  0.329483,  0.328505,  0.327536
        .s_floating  0.326575,  0.325622,  0.324678,  0.323742,  0.322814
        .s_floating  0.321894,  0.320982,  0.320078,  0.319181,  0.318293
        .s_floating  0.317412,  0.316539,  0.315673,  0.314815,  0.313965
        .s_floating  0.313122,  0.312286,  0.311457,  0.310636,  0.309821
        .s_floating  0.309014,  0.308214,  0.307421,  0.306635,  0.305856
        .s_floating  0.305083,  0.304317,  0.303558,  0.302806,  0.302060
        .s_floating  0.301321,  0.300588,  0.299862,  0.299142,  0.298429
        .s_floating  0.297721,  0.297020,  0.296326,  0.295637,  0.294955
        .s_floating  0.294278,  0.293608,  0.292944,  0.292285,  0.291633
        .s_floating  0.290986,  0.290345,  0.289710,  0.289081,  0.288457
        .s_floating  0.287839,  0.287226,  0.286620,  0.286018,  0.285422
        .s_floating  0.284832,  0.284247,  0.283668,  0.283093,  0.282524
        .s_floating  0.281961,  0.281402,  0.280849,  0.280301,  0.279758
        .s_floating  0.279220,  0.278687,  0.278160,  0.277637,  0.277119
        .s_floating  0.276606,  0.276098,  0.275595,  0.275097,  0.274604
        .s_floating  0.274115,  0.273631,  0.273152,  0.272678,  0.272208
        .s_floating  0.271743,  0.271283,  0.270827,  0.270375,  0.269928
        .s_floating  0.269486,  0.269048,  0.268615,  0.268186,  0.267762
        .s_floating  0.267341,  0.266926,  0.266514,  0.266107,  0.265704
        .s_floating  0.265306,  0.264911,  0.264521,  0.264135,  0.263753
        .s_floating  0.263376,  0.263002,  0.262633,  0.262268,  0.261906
        .s_floating  0.261549,  0.261196,  0.260847,  0.260502,  0.260161
        .s_floating  0.259823,  0.259490,  0.259161,  0.258835,  0.258513
        .s_floating  0.258196,  0.257882,  0.257572,  0.257265,  0.256963
        .s_floating  0.256664,  0.256369,  0.256078,  0.255790,  0.255507
        .s_floating  0.255226,  0.254950,  0.254677,  0.254408,  0.254143
        .s_floating  0.253881,  0.253623,  0.253368,  0.253117,  0.252869
        .s_floating  0.252625,  0.252385,  0.252148,  0.251915,  0.251685
        .s_floating  0.251458,  0.251235,  0.251016,  0.250800,  0.250587
        .s_floating  0.250378,  0.250173,  0.249970,  0.249771,  0.249576
        .s_floating  0.249384,  0.249195,  0.249010,  0.248828,  0.248649
        .s_floating  0.248473,  0.248301,  0.248133,  0.247967,  0.247805
        .s_floating  0.247646,  0.247491,  0.247338,  0.247189,  0.247043
        .s_floating  0.246901,  0.246761,  0.246625,  0.246493,  0.246363
        .s_floating  0.246236,  0.246113,  0.245993,  0.245876,  0.245763
        .s_floating  0.245652,  0.245545,  0.245441,  0.245340,  0.245242
        .s_floating  0.245147,  0.245056,  0.244968,  0.244882,  0.244800
        .s_floating  0.244721,  0.244646,  0.244573,  0.244503,  0.244437
        .s_floating  0.244373,  0.244313,  0.244256,  0.244202,  0.244151
        .s_floating  0.244103,  0.244059,  0.244017,  0.243978,  0.243943
        .s_floating  0.243911,  0.243881,  0.243855,  0.243832,  0.243812
        .s_floating  0.243795,  0.243781,  0.243770,  0.243763,  0.243758

 /*
 #--------------------------------------------------------------------
 #
 #  Convertion table, dB to linear,
 #
 #  87 dB --> 1.0
 #
 #  The just noticeable difference for a change in sensitivity of a
 #  vowel is approximately 1 dB.  Thus all amplitudes are quantized
 #  to 1 dB steps. The first 13 table values are zeroed to match the
 #  integer vocal tract model table.
 #
 #--------------------------------------------------------------------
 */

dBtoLinearTable:
        .s_floating  0.000000,  0.000000,  0.000000,  0.000000,  0.000000
        .s_floating  0.000000,  0.000000,  0.000000,  0.000000,  0.000000
        .s_floating  0.000000,  0.000000,  0.000000,  0.000200,  0.000224
        .s_floating  0.000251,  0.000282,  0.000316,  0.000355,  0.000398
        .s_floating  0.000447,  0.000501,  0.000562,  0.000631,  0.000708
        .s_floating  0.000794,  0.000891,  0.001000,  0.001122,  0.001259
        .s_floating  0.001413,  0.001585,  0.001778,  0.001995,  0.002239
        .s_floating  0.002512,  0.002818,  0.003162,  0.003548,  0.003981
        .s_floating  0.004467,  0.005012,  0.005623,  0.006310,  0.007079
        .s_floating  0.007943,  0.008913,  0.010000,  0.011220,  0.012589
        .s_floating  0.014125,  0.015849,  0.017783,  0.019953,  0.022387
        .s_floating  0.025119,  0.028184,  0.031623,  0.035481,  0.039811
        .s_floating  0.044668,  0.050119,  0.056234,  0.063096,  0.070795
        .s_floating  0.079433,  0.089125,  0.100000,  0.112202,  0.125893
        .s_floating  0.141254,  0.158489,  0.177828,  0.199526,  0.223872
        .s_floating  0.251189,  0.281838,  0.316228,  0.354813,  0.398107
        .s_floating  0.446684,  0.501187,  0.562341,  0.630957,  0.707946
        .s_floating  0.794328,  0.891251,  1.000000,  1.122018,  1.258925
        .s_floating  1.412538,  1.584893,  1.778279,  1.995262,  2.238721
        .s_floating  2.511886,  2.818383,  3.162278,  3.548134,  3.981072
        .s_floating  4.466836,  5.011872,  5.623413,  6.309573,  7.079458
        .s_floating  7.943282,  8.912509,  10.000000,  11.220185,  12.589254
        .s_floating  14.125375,  15.848932,  17.782794,  19.952623,  22.387211
        .s_floating  25.118864,  28.183829,  31.622777,  35.481339,  39.810717

 /*
 #--------------------------------------------------------------------
 #
 #  Cosine table used by function setb()
 #
 #  (Note that the just-noticable difference for a formant frequency
 #  change in a vowel is about 3%.)
 #
 #  c = cos( 2 pi f / 10,000 ) where f = 0.0, 8.0, 16.0, ..., 4992
 #
 #--------------------------------------------------------------------
 */

TwoCosineTable:
        .s_floating  2.000000,  1.999975,  1.999899,  1.999773,  1.999596
        .s_floating  1.999368,  1.999090,  1.998762,  1.998383,  1.997954
        .s_floating  1.997474,  1.996944,  1.996363,  1.995732,  1.995050
        .s_floating  1.994318,  1.993535,  1.992703,  1.991819,  1.990886
        .s_floating  1.989902,  1.988868,  1.987784,  1.986649,  1.985464
        .s_floating  1.984229,  1.982944,  1.981609,  1.980224,  1.978789
        .s_floating  1.977303,  1.975768,  1.974183,  1.972548,  1.970863
        .s_floating  1.969129,  1.967344,  1.965510,  1.963626,  1.961693
        .s_floating  1.959710,  1.957678,  1.955596,  1.953464,  1.951284
        .s_floating  1.949054,  1.946775,  1.944446,  1.942069,  1.939642
        .s_floating  1.937166,  1.934642,  1.932068,  1.929446,  1.926775
        .s_floating  1.924055,  1.921287,  1.918470,  1.915605,  1.912691
        .s_floating  1.909729,  1.906719,  1.903660,  1.900554,  1.897399
        .s_floating  1.894197,  1.890946,  1.887648,  1.884302,  1.880909
        .s_floating  1.877468,  1.873979,  1.870444,  1.866860,  1.863230
        .s_floating  1.859553,  1.855829,  1.852058,  1.848240,  1.844375
        .s_floating  1.840464,  1.836506,  1.832502,  1.828451,  1.824355
        .s_floating  1.820212,  1.816023,  1.811789,  1.807508,  1.803182
        .s_floating  1.798811,  1.794393,  1.789931,  1.785423,  1.780871
        .s_floating  1.776273,  1.771630,  1.766943,  1.762211,  1.757434
        .s_floating  1.752613,  1.747748,  1.742839,  1.737885,  1.732888
        .s_floating  1.727847,  1.722762,  1.717634,  1.712462,  1.707247
        .s_floating  1.701989,  1.696688,  1.691344,  1.685957,  1.680528
        .s_floating  1.675056,  1.669542,  1.663986,  1.658387,  1.652747
        .s_floating  1.647065,  1.641342,  1.635577,  1.629770,  1.623923
        .s_floating  1.618034,  1.612105,  1.606134,  1.600124,  1.594072
        .s_floating  1.587981,  1.581849,  1.575678,  1.569466,  1.563215
        .s_floating  1.556925,  1.550595,  1.544226,  1.537818,  1.531371
        .s_floating  1.524885,  1.518361,  1.511798,  1.505198,  1.498559
        .s_floating  1.491882,  1.485168,  1.478416,  1.471627,  1.464801
        .s_floating  1.457937,  1.451037,  1.444100,  1.437127,  1.430117
        .s_floating  1.423071,  1.415990,  1.408872,  1.401719,  1.394530
        .s_floating  1.387307,  1.380048,  1.372754,  1.365426,  1.358063
        .s_floating  1.350666,  1.343234,  1.335769,  1.328270,  1.320737
        .s_floating  1.313172,  1.305572,  1.297940,  1.290275,  1.282578
        .s_floating  1.274848,  1.267086,  1.259292,  1.251466,  1.243608
        .s_floating  1.235719,  1.227799,  1.219848,  1.211866,  1.203853
        .s_floating  1.195810,  1.187737,  1.179633,  1.171500,  1.163338
        .s_floating  1.155145,  1.146924,  1.138674,  1.130395,  1.122087
        .s_floating  1.113751,  1.105387,  1.096995,  1.088575,  1.080128
        .s_floating  1.071654,  1.063152,  1.054624,  1.046068,  1.037487
        .s_floating  1.028879,  1.020245,  1.011586,  1.002901,  0.994190
        .s_floating  0.985455,  0.976694,  0.967909,  0.959100,  0.950266
        .s_floating  0.941408,  0.932526,  0.923621,  0.914693,  0.905741
        .s_floating  0.896766,  0.887769,  0.878750,  0.869708,  0.860644
        .s_floating  0.851559,  0.842452,  0.833323,  0.824174,  0.815004
        .s_floating  0.805813,  0.796602,  0.787370,  0.778119,  0.768848
        .s_floating  0.759558,  0.750249,  0.740920,  0.731573,  0.722208
        .s_floating  0.712824,  0.703422,  0.694002,  0.684565,  0.675111
        .s_floating  0.665639,  0.656151,  0.646646,  0.637125,  0.627587
        .s_floating  0.618034,  0.608465,  0.598881,  0.589282,  0.579667
        .s_floating  0.570039,  0.560395,  0.550738,  0.541066,  0.531381
        .s_floating  0.521683,  0.511971,  0.502247,  0.492510,  0.482760
        .s_floating  0.472998,  0.463224,  0.453439,  0.443642,  0.433833
        .s_floating  0.424014,  0.414184,  0.404344,  0.394493,  0.384633
        .s_floating  0.374763,  0.364883,  0.354994,  0.345096,  0.335189
        .s_floating  0.325274,  0.315351,  0.305420,  0.295481,  0.285534
        .s_floating  0.275581,  0.265620,  0.255653,  0.245679,  0.235699
        .s_floating  0.225713,  0.215721,  0.205724,  0.195722,  0.185714
        .s_floating  0.175702,  0.165686,  0.155665,  0.145641,  0.135613
        .s_floating  0.125581,  0.115546,  0.105509,  0.095468,  0.085425
        .s_floating  0.075380,  0.065334,  0.055285,  0.045235,  0.035184
        .s_floating  0.025132,  0.015080,  0.005027,  -0.005027,  -0.015080
        .s_floating  -0.025132,  -0.035184,  -0.045235,  -0.055285,  -0.065334
        .s_floating  -0.075380,  -0.085425,  -0.095468,  -0.105509,  -0.115546
        .s_floating  -0.125581,  -0.135613,  -0.145641,  -0.155665,  -0.165686
        .s_floating  -0.175702,  -0.185714,  -0.195722,  -0.205724,  -0.215721
        .s_floating  -0.225713,  -0.235699,  -0.245679,  -0.255653,  -0.265620
        .s_floating  -0.275581,  -0.285534,  -0.295481,  -0.305420,  -0.315351
        .s_floating  -0.325274,  -0.335189,  -0.345096,  -0.354994,  -0.364883
        .s_floating  -0.374763,  -0.384633,  -0.394493,  -0.404344,  -0.414184
        .s_floating  -0.424014,  -0.433833,  -0.443642,  -0.453439,  -0.463224
        .s_floating  -0.472998,  -0.482760,  -0.492510,  -0.502247,  -0.511971
        .s_floating  -0.521683,  -0.531381,  -0.541066,  -0.550738,  -0.560395
        .s_floating  -0.570039,  -0.579667,  -0.589282,  -0.598881,  -0.608465
        .s_floating  -0.618034,  -0.627587,  -0.637125,  -0.646646,  -0.656151
        .s_floating  -0.665639,  -0.675111,  -0.684565,  -0.694002,  -0.703422
        .s_floating  -0.712824,  -0.722208,  -0.731573,  -0.740920,  -0.750249
        .s_floating  -0.759558,  -0.768848,  -0.778119,  -0.787370,  -0.796602
        .s_floating  -0.805813,  -0.815004,  -0.824174,  -0.833323,  -0.842452
        .s_floating  -0.851559,  -0.860644,  -0.869708,  -0.878750,  -0.887769
        .s_floating  -0.896766,  -0.905741,  -0.914693,  -0.923621,  -0.932526
        .s_floating  -0.941408,  -0.950266,  -0.959100,  -0.967909,  -0.976694
        .s_floating  -0.985455,  -0.994190,  -1.002901,  -1.011586,  -1.020245
        .s_floating  -1.028879,  -1.037487,  -1.046068,  -1.054624,  -1.063152
        .s_floating  -1.071654,  -1.080128,  -1.088575,  -1.096995,  -1.105387
        .s_floating  -1.113751,  -1.122087,  -1.130395,  -1.138674,  -1.146924
        .s_floating  -1.155145,  -1.163338,  -1.171500,  -1.179633,  -1.187737
        .s_floating  -1.195810,  -1.203853,  -1.211866,  -1.219848,  -1.227799
        .s_floating  -1.235719,  -1.243608,  -1.251466,  -1.259292,  -1.267086
        .s_floating  -1.274848,  -1.282578,  -1.290275,  -1.297940,  -1.305572
        .s_floating  -1.313172,  -1.320737,  -1.328270,  -1.335769,  -1.343234
        .s_floating  -1.350666,  -1.358063,  -1.365426,  -1.372754,  -1.380048
        .s_floating  -1.387307,  -1.394530,  -1.401719,  -1.408872,  -1.415990
        .s_floating  -1.423071,  -1.430117,  -1.437127,  -1.444100,  -1.451037
        .s_floating  -1.457937,  -1.464801,  -1.471627,  -1.478416,  -1.485168
        .s_floating  -1.491882,  -1.498559,  -1.505198,  -1.511798,  -1.518361
        .s_floating  -1.524885,  -1.531371,  -1.537818,  -1.544226,  -1.550595
        .s_floating  -1.556925,  -1.563215,  -1.569466,  -1.575678,  -1.581849
        .s_floating  -1.587981,  -1.594072,  -1.600124,  -1.606134,  -1.612105
        .s_floating  -1.618034,  -1.623923,  -1.629770,  -1.635577,  -1.641342
        .s_floating  -1.647065,  -1.652747,  -1.658387,  -1.663986,  -1.669542
        .s_floating  -1.675056,  -1.680528,  -1.685957,  -1.691344,  -1.696688
        .s_floating  -1.701989,  -1.707247,  -1.712462,  -1.717634,  -1.722762
        .s_floating  -1.727847,  -1.732888,  -1.737885,  -1.742839,  -1.747748
        .s_floating  -1.752613,  -1.757434,  -1.762211,  -1.766943,  -1.771630
        .s_floating  -1.776273,  -1.780871,  -1.785423,  -1.789931,  -1.794393
        .s_floating  -1.798811,  -1.803182,  -1.807508,  -1.811789,  -1.816023
        .s_floating  -1.820212,  -1.824355,  -1.828451,  -1.832502,  -1.836506
        .s_floating  -1.840464,  -1.844375,  -1.848240,  -1.852058,  -1.855829
        .s_floating  -1.859553,  -1.863230,  -1.866860,  -1.870444,  -1.873979
        .s_floating  -1.877468,  -1.880909,  -1.884302,  -1.887648,  -1.890946
        .s_floating  -1.894197,  -1.897399,  -1.900554,  -1.903660,  -1.906719
        .s_floating  -1.909729,  -1.912691,  -1.915605,  -1.918470,  -1.921287
        .s_floating  -1.924055,  -1.926775,  -1.929446,  -1.932068,  -1.934642
        .s_floating  -1.937166,  -1.939642,  -1.942069,  -1.944446,  -1.946775
        .s_floating  -1.949054,  -1.951284,  -1.953464,  -1.955596,  -1.957678
        .s_floating  -1.959710,  -1.961693,  -1.963626,  -1.965510,  -1.967344
        .s_floating  -1.969129,  -1.970863,  -1.972548,  -1.974183,  -1.975768
        .s_floating  -1.977303,  -1.978789,  -1.980224,  -1.981609,  -1.982944
        .s_floating  -1.984229,  -1.985464,  -1.986649,  -1.987784,  -1.988868
        .s_floating  -1.989902,  -1.990886,  -1.991819,  -1.992703,  -1.993535
        .s_floating  -1.994318,  -1.995050,  -1.995732,  -1.996363,  -1.996944
        .s_floating  -1.997474,  -1.997954,  -1.998383,  -1.998762,  -1.999090
        .s_floating  -1.999368,  -1.999596,  -1.999773,  -1.999899,  -1.999975

 /*
 #--------------------------------------------------------------------
 #
 #  Radius table used by function setc()
 #
 #  radius = exp( - pi * b /10,000 ) where b = 0.0, 8.0, ..., 4992
 #
 #--------------------------------------------------------------------
 */

RadiusTable:
        .s_floating  0.0, 0.997490,  0.994986,  0.992489,  0.989997,  0.987512
        .s_floating  0.985033,  0.982561,  0.980095,  0.977634,  0.975180
        .s_floating  0.972733,  0.970291,  0.967855,  0.965426,  0.963003
        .s_floating  0.960585,  0.958174,  0.955769,  0.953370,  0.950977
        .s_floating  0.948590,  0.946209,  0.943834,  0.941465,  0.939101
        .s_floating  0.936744,  0.934393,  0.932047,  0.929708,  0.927374
        .s_floating  0.925046,  0.922724,  0.920408,  0.918098,  0.915793
        .s_floating  0.913495,  0.911202,  0.908914,  0.906633,  0.904357
        .s_floating  0.902087,  0.899823,  0.897564,  0.895311,  0.893064
        .s_floating  0.890822,  0.888586,  0.886356,  0.884131,  0.881911
        .s_floating  0.879698,  0.877490,  0.875287,  0.873090,  0.870898
        .s_floating  0.868712,  0.866532,  0.864357,  0.862187,  0.860023
        .s_floating  0.857864,  0.855711,  0.853563,  0.851420,  0.849283
        .s_floating  0.847151,  0.845025,  0.842904,  0.840788,  0.838677
        .s_floating  0.836572,  0.834472,  0.832378,  0.830288,  0.828204
        .s_floating  0.826125,  0.824052,  0.821983,  0.819920,  0.817862
        .s_floating  0.815809,  0.813761,  0.811718,  0.809681,  0.807649
        .s_floating  0.805621,  0.803599,  0.801582,  0.799570,  0.797563
        .s_floating  0.795561,  0.793564,  0.791572,  0.789585,  0.787603
        .s_floating  0.785626,  0.783654,  0.781687,  0.779725,  0.777768
        .s_floating  0.775815,  0.773868,  0.771926,  0.769988,  0.768055
        .s_floating  0.766127,  0.764204,  0.762286,  0.760372,  0.758464
        .s_floating  0.756560,  0.754661,  0.752767,  0.750877,  0.748992
        .s_floating  0.747112,  0.745237,  0.743366,  0.741500,  0.739639
        .s_floating  0.737783,  0.735931,  0.734083,  0.732241,  0.730403
        .s_floating  0.728569,  0.726740,  0.724916,  0.723097,  0.721282
        .s_floating  0.719471,  0.717665,  0.715864,  0.714067,  0.712274
        .s_floating  0.710487,  0.708703,  0.706924,  0.705150,  0.703380
        .s_floating  0.701614,  0.699853,  0.698096,  0.696344,  0.694596
        .s_floating  0.692853,  0.691113,  0.689379,  0.687648,  0.685922
        .s_floating  0.684200,  0.682483,  0.680770,  0.679061,  0.677357
        .s_floating  0.675656,  0.673960,  0.672269,  0.670581,  0.668898
        .s_floating  0.667219,  0.665544,  0.663873,  0.662207,  0.660545
        .s_floating  0.658887,  0.657233,  0.655583,  0.653938,  0.652296
        .s_floating  0.650659,  0.649026,  0.647396,  0.645771,  0.644150
        .s_floating  0.642534,  0.640921,  0.639312,  0.637707,  0.636106
        .s_floating  0.634510,  0.632917,  0.631328,  0.629744,  0.628163
        .s_floating  0.626586,  0.625013,  0.623444,  0.621880,  0.620319
        .s_floating  0.618762,  0.617208,  0.615659,  0.614114,  0.612572
        .s_floating  0.611035,  0.609501,  0.607971,  0.606445,  0.604923
        .s_floating  0.603404,  0.601890,  0.600379,  0.598872,  0.597368
        .s_floating  0.595869,  0.594373,  0.592881,  0.591393,  0.589909
        .s_floating  0.588428,  0.586951,  0.585478,  0.584008,  0.582542
        .s_floating  0.581080,  0.579621,  0.578166,  0.576715,  0.575267
        .s_floating  0.573823,  0.572383,  0.570946,  0.569513,  0.568084
        .s_floating  0.566658,  0.565235,  0.563816,  0.562401,  0.560990
        .s_floating  0.559581,  0.558177,  0.556776,  0.555378,  0.553984
        .s_floating  0.552593,  0.551206,  0.549823,  0.548443,  0.547066
        .s_floating  0.545693,  0.544323,  0.542957,  0.541594,  0.540234
        .s_floating  0.538878,  0.537526,  0.536176,  0.534831,  0.533488
        .s_floating  0.532149,  0.530813,  0.529481,  0.528152,  0.526826
        .s_floating  0.525504,  0.524185,  0.522869,  0.521556,  0.520247
        .s_floating  0.518941,  0.517639,  0.516339,  0.515043,  0.513750
        .s_floating  0.512461,  0.511175,  0.509891,  0.508612,  0.507335
        .s_floating  0.506061,  0.504791,  0.503524,  0.502260,  0.500999
        .s_floating  0.499742,  0.498487,  0.497236,  0.495988,  0.494743
        .s_floating  0.493501,  0.492262,  0.491027,  0.489794,  0.488565
        .s_floating  0.487338,  0.486115,  0.484895,  0.483678,  0.482464
        .s_floating  0.481253,  0.480045,  0.478840,  0.477638,  0.476439
        .s_floating  0.475243,  0.474050,  0.472860,  0.471673,  0.470489
        .s_floating  0.469308,  0.468130,  0.466955,  0.465783,  0.464614
        .s_floating  0.463448,  0.462284,  0.461124,  0.459966,  0.458812
        .s_floating  0.457660,  0.456511,  0.455366,  0.454223,  0.453082
        .s_floating  0.451945,  0.450811,  0.449679,  0.448550,  0.447424
        .s_floating  0.446301,  0.445181,  0.444064,  0.442949,  0.441837
        .s_floating  0.440728,  0.439622,  0.438518,  0.437417,  0.436320
        .s_floating  0.435224,  0.434132,  0.433042,  0.431955,  0.430871
        .s_floating  0.429789,  0.428711,  0.427634,  0.426561,  0.425490
        .s_floating  0.424422,  0.423357,  0.422294,  0.421234,  0.420177
        .s_floating  0.419122,  0.418070,  0.417021,  0.415974,  0.414930
        .s_floating  0.413888,  0.412849,  0.411813,  0.410779,  0.409748
        .s_floating  0.408720,  0.407694,  0.406670,  0.405650,  0.404631
        .s_floating  0.403616,  0.402603,  0.401592,  0.400584,  0.399578
        .s_floating  0.398576,  0.397575,  0.396577,  0.395582,  0.394589
        .s_floating  0.393598,  0.392610,  0.391625,  0.390642,  0.389661
        .s_floating  0.388683,  0.387707,  0.386734,  0.385763,  0.384795
        .s_floating  0.383829,  0.382866,  0.381905,  0.380946,  0.379990
        .s_floating  0.379036,  0.378085,  0.377136,  0.376189,  0.375245
        .s_floating  0.374303,  0.373363,  0.372426,  0.371491,  0.370559
        .s_floating  0.369629,  0.368701,  0.367775,  0.366852,  0.365931
        .s_floating  0.365013,  0.364097,  0.363183,  0.362271,  0.361362
        .s_floating  0.360455,  0.359550,  0.358647,  0.357747,  0.356849
        .s_floating  0.355953,  0.355060,  0.354169,  0.353280,  0.352393
        .s_floating  0.351508,  0.350626,  0.349746,  0.348868,  0.347992
        .s_floating  0.347119,  0.346247,  0.345378,  0.344511,  0.343647
        .s_floating  0.342784,  0.341924,  0.341065,  0.340209,  0.339355
        .s_floating  0.338503,  0.337654,  0.336806,  0.335961,  0.335117
        .s_floating  0.334276,  0.333437,  0.332600,  0.331765,  0.330933
        .s_floating  0.330102,  0.329273,  0.328447,  0.327622,  0.326800
        .s_floating  0.325980,  0.325161,  0.324345,  0.323531,  0.322719
        .s_floating  0.321909,  0.321101,  0.320295,  0.319491,  0.318689
        .s_floating  0.317889,  0.317091,  0.316295,  0.315501,  0.314709
        .s_floating  0.313919,  0.313131,  0.312345,  0.311561,  0.310779
        .s_floating  0.309999,  0.309221,  0.308445,  0.307671,  0.306898
        .s_floating  0.306128,  0.305360,  0.304593,  0.303828,  0.303066
        .s_floating  0.302305,  0.301546,  0.300789,  0.300034,  0.299281
        .s_floating  0.298530,  0.297781,  0.297033,  0.296288,  0.295544
        .s_floating  0.294802,  0.294062,  0.293324,  0.292588,  0.291853
        .s_floating  0.291121,  0.290390,  0.289661,  0.288934,  0.288209
        .s_floating  0.287485,  0.286764,  0.286044,  0.285326,  0.284610
        .s_floating  0.283895,  0.283183,  0.282472,  0.281763,  0.281055
        .s_floating  0.280350,  0.279646,  0.278944,  0.278244,  0.277546
        .s_floating  0.276849,  0.276154,  0.275461,  0.274769,  0.274080
        .s_floating  0.273392,  0.272706,  0.272021,  0.271338,  0.270657
        .s_floating  0.269978,  0.269300,  0.268624,  0.267950,  0.267277
        .s_floating  0.266606,  0.265937,  0.265270,  0.264604,  0.263940
        .s_floating  0.263277,  0.262616,  0.261957,  0.261299,  0.260644
        .s_floating  0.259989,  0.259337,  0.258686,  0.258036,  0.257389
        .s_floating  0.256743,  0.256098,  0.255455,  0.254814,  0.254174
        .s_floating  0.253536,  0.252900,  0.252265,  0.251632,  0.251000
        .s_floating  0.250370,  0.249742,  0.249115,  0.248490,  0.247866
        .s_floating  0.247244,  0.246623,  0.246004,  0.245387,  0.244771
        .s_floating  0.244156,  0.243543,  0.242932,  0.242322,  0.241714
        .s_floating  0.241107,  0.240502,  0.239898,  0.239296,  0.238696
        .s_floating  0.238096,  0.237499,  0.236903,  0.236308,  0.235715
        .s_floating  0.235123,  0.234533,  0.233944,  0.233357,  0.232771
        .s_floating  0.232187,  0.231604,  0.231023,  0.230443,  0.229864
        .s_floating  0.229287,  0.228712,  0.228138,  0.227565,  0.226994
        .s_floating  0.226424,  0.225856,  0.225289,  0.224723,  0.224159
        .s_floating  0.223597,  0.223035,  0.222476,  0.221917,  0.221360
        .s_floating  0.220804,  0.220250,  0.219697,  0.219146,  0.218596
        .s_floating  0.218047,  0.217500,  0.216954,  0.216409,  0.215866
        .s_floating  0.215324,  0.214784,  0.214245,  0.213707,  0.213170
        .s_floating  0.212635,  0.212102,  0.211569,  0.211038,  0.210508
        .s_floating  0.209980,  0.209453,  0.208927,  0.208403
